/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.i18n;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.net.MalformedURLException;
import java.nio.file.Paths;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.stream.Collectors;

import org.junit.Test;

import com.google.common.collect.ImmutableMap;

public class EnumTranslatorTest {

  public static final Locale NZ = new Locale("en", "NZ");

  enum TestEnum {
    VALUE1,
    VALUE2,
    VALUE3
  }

  I18nClassLoader classLoader = new I18nClassLoader();
  RiskscapeMessageSource messages = new RiskscapeMessageSource("messages", classLoader);
  TranslationContext context = new TranslationContext(NZ, messages);
  EnumTranslator subject = new EnumTranslator(context);

  private ResourceClassLoader newResourceClassLoader(String pathSegment) throws MalformedURLException {
    return new ResourceClassLoader(
        "nz/org/riskscape/engine/i18n/" + pathSegment + "/",
        Paths.get("src", "test", "resources").toUri().toURL()
    );
  }

  private void init(String pathSegment) throws MalformedURLException {
    classLoader.append(pathSegment, newResourceClassLoader(pathSegment));
  }

  @Test
  public void canDisplayEnumLabels() throws Exception {
    init("enum");
    List<String> translated = subject.getEntries(TestEnum.class).stream()
        .map(entry -> entry.getLabel())
        .collect(Collectors.toList());

    assertThat(translated, contains(
        equalTo("this is value 1"),
        equalTo("value2"), // i18n message is missing - default to non-shouty enum name
        equalTo("this is value 3")
    ));
  }

  @Test
  public void canDisplayDefaultForEnumLabels() throws Exception {
    init("enum");
    // we support fallback abels for enums, where the canonical enum name might not make
    // sense to the user. E.g. BooleanExpressionAsker.Comparison where the symbol
    // doesn't need translation and displaying ">=" is nicer than displaying "GREATER_THAN_EQUAL"
    Map<TestEnum, String> defaultLabels = ImmutableMap.of(
        TestEnum.VALUE1, "not this",
        TestEnum.VALUE2, "some default",
        TestEnum.VALUE3, "nor this"
    );
    List<String> translated = subject.getEntries(TestEnum.class).stream()
        .map(entry -> entry.getLabelOr(defaultLabels.get(entry.getValue())))
        .collect(Collectors.toList());

    assertThat(translated, contains(
        equalTo("this is value 1"),
        equalTo("some default"), // used the defaultLabel for the missing key this time
        equalTo("this is value 3")
    ));
  }

  @Test
  public void canDisplayEnumDescription() throws Exception {
    init("enum");
    List<String> translated = subject.getEntries(TestEnum.class).stream()
        .map(entry -> entry.getDescription())
        .collect(Collectors.toList());

    assertThat(translated, contains(
        equalTo("this is the best enum value ever"),
        equalTo("this one is also good"),
        equalTo(null)
    ));
  }
}
