/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.geo;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.io.InputStreamReader;

import org.junit.Before;
import org.junit.Test;
import org.locationtech.jts.geom.Envelope;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.io.ParseException;
import org.locationtech.jts.io.WKTReader;

/**
 * Does some quite specific unit tests on the compexity heuristic
 */
public class IntersectionIndexComplexityTest {

  IntersectionIndex.Options options = IntersectionIndex.defaultOptions();
  Geometry geometry;
  Envelope envelope;

  // various facts about the the known geometry to make the test more readable
  long morePoints = 400;
  long lessPoints = 300;
  double largerRatio = 0.5;
  double smallerRatio = 0.25;
  long biggerSize = 3000;
  long smallerSize = 1800;

  @Before
  public void setup() throws ParseException {
    geometry = getSa1Feature();
    envelope = geometry.getEnvelopeInternal();

    // set the defaults so they never take effect, e.g. nothing should be complex
    options.setCutRatio(1.1D);
    options.setCutPoints(Integer.MAX_VALUE);
    options.setCutRatioPoints(Integer.MAX_VALUE);
  }

  @Test
  public void sanityCheckExampleGeometry() throws Exception {

    // some sanity checks that we've got the right feature
    assertThat(geometry.getNumPoints(), equalTo(387));
    // roughly 29 percent
    assertThat((int) (geometry.getArea() / envelope.getArea() * 100), equalTo(29));
    // the average of these two gives our 'size'
    assertThat(Math.round(envelope.getWidth()), equalTo(2004L));
    assertThat(Math.round(envelope.getHeight()), equalTo(2752L));



    assertFalse(options.isTooComplex(geometry, envelope));
  }

  @Test
  public void tooComplexWhenPointsExceededRegardlessOfRatio() throws Exception {
    options.setCutPoints(lessPoints);
    assertTrue(options.isTooComplex(geometry, envelope));
  }

  @Test
  public void tooComplexIfAreaLessThanRatio() {
    options.setCutRatioPoints(lessPoints);
    options.setCutRatio(largerRatio);
    assertTrue(options.isTooComplex(geometry, envelope));
  }

  @Test
  public void notComplexIfAreaGreaterThanRatio() {
    options.setCutRatioPoints(lessPoints);
    options.setCutRatio(smallerRatio);
    assertFalse(options.isTooComplex(geometry, envelope));
  }

  @Test
  public void notComplexWhenUnderMinPoints() {
    // ratio qualifies, but not enough points
    options.setCutRatioPoints(morePoints);
    options.setCutRatio(largerRatio);
    assertFalse(options.isTooComplex(geometry, envelope));
  }

  @Test
  public void notComplexWhenSizeSmallerThanCutSize() throws Exception {
    options.setCutSizeMapUnits(biggerSize);
    assertFalse(options.isTooComplex(geometry, envelope));
  }

  @Test
  public void tooComplexWhenSizeBiggerThenCutSize() throws Exception {
    options.setCutSizeMapUnits(smallerSize);
    assertTrue(options.isTooComplex(geometry, envelope));
  }

  @Test
  public void squaresAreNeverTooComplexEvenIfLargerThanCutSize() throws Exception {
    geometry = getSquare();
    envelope = geometry.getEnvelopeInternal();

    // sanity check I defined a square
    assertThat(geometry.getArea(), equalTo(1D));
    assertThat(envelope.getArea(), equalTo(1D));

    // these should mean any geometry is too complex - warning, don't set these values or it'll crash
    options.setCutPoints(0);
    options.setCutRatio(0);
    options.setCutRatioPoints(0);
    options.setCutSizeMapUnits(0);

    assertFalse(options.isTooComplex(geometry, envelope));
  }

  public Geometry getSa1Feature() throws ParseException {
    // a fairly interestingly shaped sa1 in karori
    final InputStreamReader wkt = new InputStreamReader(getClass().getResourceAsStream("sa1-in-karori.wkt"));

    return new WKTReader().read(wkt);
  }

  public Geometry getSquare() throws ParseException {
    return new WKTReader().read("""
        POLYGON (( 0 0, 0 1, 1 1, 1 0, 0 0 ))
        """);
  }
}
