/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.geo;

import static org.mockito.ArgumentMatchers.*;
import static org.mockito.Mockito.*;

import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

import org.junit.Before;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.LineString;
import org.locationtech.jts.geom.Point;
import org.locationtech.jts.geom.Polygon;
import org.geotools.api.referencing.crs.CoordinateReferenceSystem;

import nz.org.riskscape.engine.ProjectTest;
import nz.org.riskscape.engine.geo.ProjectGeometryOp.Projected;
import nz.org.riskscape.engine.gt.LatLongGeometryHelper;
import nz.org.riskscape.engine.gt.NZMGGeometryHelper;
import nz.org.riskscape.engine.gt.NewYorkFootGeometryHelper;

public abstract class GeometryMeasureTest extends ProjectTest {

  NZMGGeometryHelper nzmgHelper = new NZMGGeometryHelper(project.getSridSet());
  LatLongGeometryHelper latLongHelper = new LatLongGeometryHelper(project.getSridSet());
  NewYorkFootGeometryHelper newYorkFootGeomHelper = new NewYorkFootGeometryHelper(project.getSridSet());

  ProjectGeometryOp projectOpMock = mock(ProjectGeometryOp.class);

  public Point point = nzmgHelper.point(1, 1);
  public Point empty = nzmgHelper.emptyPoint();

  public LineString lineString = nzmgHelper.line(new Coordinate[]{
    new Coordinate(1, 1),
    new Coordinate(2, 1),
    new Coordinate(2, 2)
  });

  public Polygon simplePolygon = nzmgHelper.box(new Coordinate[]{
    new Coordinate(1, 1),
    new Coordinate(2, 1),
    new Coordinate(2, 2),
    new Coordinate(1, 2),
    new Coordinate(1, 1)
  });

  public Polygon complexPolygon = nzmgHelper.box(
      nzmgHelper.ring(
          new Coordinate[]{
            new Coordinate(0, 0),
            new Coordinate(3, 0),
            new Coordinate(3, 3),
            new Coordinate(0, 3),
            new Coordinate(0, 0)
          }
      ),
      nzmgHelper.ring(
          new Coordinate[]{
            new Coordinate(1, 1),
            new Coordinate(2, 1),
            new Coordinate(2, 2),
            new Coordinate(1, 2),
            new Coordinate(1, 1)
          }
      )
  );

  @Before
  public void setupMocks() {
    primeMock(point, nzmgHelper.getCrs());
    primeMock(lineString, nzmgHelper.getCrs());
    primeMock(simplePolygon, nzmgHelper.getCrs());
    primeMock(complexPolygon, nzmgHelper.getCrs());

  }

  <G extends Geometry> G primeMock(G geom, CoordinateReferenceSystem crs) {
    primeMock(geom, crs, crs, geom);
    return geom;
  }

  private Projected mockProjected(Geometry input, CoordinateReferenceSystem inputCrs,
      CoordinateReferenceSystem producedCrs, Geometry produced) {
    Projected mockProjected = mock(Projected.class);
    when(mockProjected.getSource()).thenReturn(input);
    when(mockProjected.getSourceCrs()).thenReturn(inputCrs);
    when(mockProjected.getProjected()).thenReturn(produced);
    when(mockProjected.getProjectedCrs()).thenReturn(producedCrs);
    return mockProjected;
  }

  <G extends Geometry> G primeMock(G input, CoordinateReferenceSystem inputCrs,
      CoordinateReferenceSystem producedCrs, Geometry... produced) {
    List<Projected> projected = Arrays.stream(produced)
        .map(g -> mockProjected(input, inputCrs, producedCrs, g))
        .collect(Collectors.toList());
    when(projectOpMock.apply(eq(input))).thenReturn(projected);
    return input;
  }

}
