/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.geo;


import static org.geotools.measure.Units.*;
import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import static systems.uom.common.USCustomary.*;

import org.junit.Test;
import org.locationtech.jts.geom.LineString;



public class GeometryLengthOpTest extends GeometryMeasureTest {

  GeometryLengthOp subject = new GeometryLengthOp(project.getSridSet(), projectOpMock);

  @Test
  public void measureLengthOfAPointIsZero() throws Exception {
    assertThat(subject.apply(point, METRE), is(0.0D));
  }

  @Test
  public void measureLengthOfALineIsDistanceBetweenAllPoints() throws Exception {
    assertThat(subject.apply(lineString, METRE), is(2.0D));
  }

  @Test
  public void measureLengthOfAPolygonIsTheDistanceAroundThePerimeterOfShellAndRings() throws Exception {
    assertThat(subject.apply(simplePolygon, METRE), is(4.0D));
    assertThat(subject.apply(complexPolygon, METRE), is(16.0D));
  }

  @Test
  public void canMeasureLengthInMetricFromImperial() {
    // we start with an NZ line, because meters are easy to understand
    LineString newYorkLine = primeMock(newYorkFootGeomHelper.line(0, 0, 0, 1000), newYorkFootGeomHelper.getCrs());
    assertThat(newYorkLine.getLength(), is(1000.0D));  //sanity check

    assertThat(subject.apply(newYorkLine, METRE), closeTo(304.8D, 0.01D));  // 1000ft is approx 305 metres
  }

  @Test
  public void combinesLengthFromManyProjectedParts() {
    // north south line height of NZ
    LineString line = primeMock(latLongHelper.line(-34, 170, -47, 170), latLongHelper.getCrs(), nzmgHelper.getCrs(),
        nzmgHelper.line(0, 0, 0, 1000), nzmgHelper.line(100, 0, 100, 2000));
    assertThat(subject.apply(line, METRE),
        is(1000D + 2000D));
  }

  @Test
  public void returnsMeasureInDesiredUnits() {
    LineString box = nzmgHelper.line(0, 0, 0, 1000);

    primeMock(box, nzmgHelper.getCrs());
    assertThat(subject.apply(box, METRE), is(1000D));
    assertThat(subject.apply(box, METRE.multiply(1000)), is(1D));
    assertThat(subject.apply(box, METRE.divide(1000)), is(1000D * 1000D));


    assertThat(subject.apply(box, INCH), closeTo(39370.07D, 0.1D));
  }
}
