/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.geo;

import static nz.org.riskscape.engine.geo.GeometryFamily.*;

import static org.junit.Assert.*;

import org.junit.Test;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.LineString;
import org.locationtech.jts.geom.MultiLineString;
import org.locationtech.jts.geom.MultiPoint;
import org.locationtech.jts.geom.MultiPolygon;
import org.locationtech.jts.geom.Point;
import org.locationtech.jts.geom.Polygon;

import nz.org.riskscape.engine.ProjectTest;
import nz.org.riskscape.engine.gt.LatLongGeometryHelper;

public class GeometryFamilyTest extends ProjectTest {

  LatLongGeometryHelper latLongGeomHelper = new LatLongGeometryHelper(project.getSridSet());

  @Test
  public void getsExpectedFamily() {
    assertEquals(PUNTAL, GeometryFamily.from(latLongGeomHelper.point(1, 2)));
    assertEquals(PUNTAL, GeometryFamily.from(latLongGeomHelper.multiPoint(latLongGeomHelper.point(1, 2))));
    assertEquals(PUNTAL, GeometryFamily.fromClass(Point.class));
    assertEquals(PUNTAL, GeometryFamily.fromClass(MultiPoint.class));

    assertEquals(LINEAL, GeometryFamily.from(latLongGeomHelper.line(1, 2, 2, 2)));
    assertEquals(LINEAL, GeometryFamily.from(latLongGeomHelper.multiLine(latLongGeomHelper.line(1, 2, 2, 2))));
    assertEquals(LINEAL, GeometryFamily.fromClass(LineString.class));
    assertEquals(LINEAL, GeometryFamily.fromClass(MultiLineString.class));

    assertEquals(POLYGONAL, GeometryFamily.from(latLongGeomHelper.box(0, 0, 10, 10)));
    assertEquals(POLYGONAL, GeometryFamily.from(latLongGeomHelper.multiBox(latLongGeomHelper.box(0, 0, 10, 10))));
    assertEquals(POLYGONAL, GeometryFamily.fromClass(Polygon.class));
    assertEquals(POLYGONAL, GeometryFamily.fromClass(MultiPolygon.class));

    assertNull(GeometryFamily.fromClass(Geometry.class));   // not a specific geometry type
    assertNull(GeometryFamily.fromClass(Integer.class));    // not a geometry at all
  }

  @Test
  public void detectsGeometryInFamily() {
    assertTrue(PUNTAL.isSameFamily(latLongGeomHelper.point(1, 2)));
    assertTrue(PUNTAL.isSameFamily(latLongGeomHelper.multiPoint(latLongGeomHelper.point(1, 2))));
    assertFalse(LINEAL.isSameFamily(latLongGeomHelper.point(1, 2)));
    assertFalse(LINEAL.isSameFamily(latLongGeomHelper.multiPoint(latLongGeomHelper.point(1, 2))));
    assertFalse(POLYGONAL.isSameFamily(latLongGeomHelper.point(1, 2)));
    assertFalse(POLYGONAL.isSameFamily(latLongGeomHelper.multiPoint(latLongGeomHelper.point(1, 2))));
  }

}
