/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.geo;


import static org.hamcrest.Matchers.*;

import org.junit.Test;

import static org.junit.Assert.*;

import static systems.uom.common.USCustomary.*;
import static tech.units.indriya.unit.Units.*;

import javax.measure.quantity.Area;

import org.locationtech.jts.geom.Polygon;

import systems.uom.common.historic.CGS;

public class GeometryAreaOpTest extends GeometryMeasureTest {

  GeometryAreaOp subject = new GeometryAreaOp(project.getSridSet(), projectOpMock);

  @Test
  public void measureAreaOfAPointIsZero() throws Exception {
    assertThat(subject.apply(point, SQUARE_METRE), is(0.0D));
  }

  @Test
  public void measureAreaOfALineIsZero() throws Exception {
    assertThat(subject.apply(lineString, SQUARE_METRE), is(0.0D));
  }

  @Test
  public void measureAreaOfAPolygonIsTheAreaOfTheShellMinusRings() throws Exception {
    assertThat(subject.apply(simplePolygon, SQUARE_METRE), is(1.0D));
    assertThat(subject.apply(complexPolygon, SQUARE_METRE), is(8.0D));
  }

  @Test
  public void canMeasureAreaInMetricFromImperial() {
    Polygon newYorkBox = newYorkFootGeomHelper.box(0, 0, 1000, 1000);
    assertThat(newYorkBox.getArea(), is(1000000.0D));  //sanity check area in square feet

    primeMock(newYorkBox, newYorkFootGeomHelper.getCrs());
    // The new york box is approx 305 x 305m or 93000 square metres
    assertThat(subject.apply(newYorkBox, SQUARE_METRE), closeTo(92903.415D, 0.01D));
  }

  @Test
  public void combinesAreasFromManyProjectedParts() {
    // approximately a bbox around NZ
    Polygon box = primeMock(latLongHelper.box(-47, 166, -34, 178), latLongHelper.getCrs(),
        nzmgHelper.getCrs(), nzmgHelper.box(1000, 1000, 2000, 2000), nzmgHelper.box(4000, 4000, 5000, 5000));
    assertThat(subject.apply(box, SQUARE_METRE),
        is(1000D * 1000D * 2)); // two boxes that are 1000 x 1000
  }

  @Test
  public void returnsMeasureInDesiredUnits() {
    Polygon box = nzmgHelper.box(0, 0, 1000, 1000);

    primeMock(box, nzmgHelper.getCrs());
    assertThat(subject.apply(box, SQUARE_METRE), is(1000D * 1000D));
    assertThat(subject.apply(box, HECTARE), is(100D));

    assertThat(subject.apply(box, CGS.CENTIMETRE.multiply(CGS.CENTIMETRE).asType(Area.class)),
        is(1000D * 1000D * 100 * 100));
  }
}
