/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.geo;

import static nz.org.riskscape.engine.GeoHelper.*;
import static org.junit.Assert.*;
import static org.mockito.Mockito.*;

import org.hamcrest.Matcher;
import org.junit.Test;
import org.locationtech.jts.geom.Geometry;

import nz.org.riskscape.engine.SRIDSet;
import nz.org.riskscape.engine.gt.NZTMGeometryHelper;

public class GeoHelperTest {

  SRIDSet sridSet = spy(new SRIDSet());
  NZTMGeometryHelper nztmHelper = new NZTMGeometryHelper(sridSet);

  @Test
  public void matchesGeometryWithinTolerance() {
    Matcher<Geometry> matcher = geometryMatch(nztmHelper.box(1, 1, 10, 10), METER_TOLERANCE_NEAREST_MM);

    assertTrue(matcher.matches(nztmHelper.box(1.00095, 1, 10, 10)));
    assertTrue(matcher.matches(nztmHelper.box(1, 1.00095, 10, 10)));
    assertTrue(matcher.matches(nztmHelper.box(1, 1, 10.00095, 10)));
    assertTrue(matcher.matches(nztmHelper.box(1, 1, 10, 10.00095)));
  }

  @Test
  public void failsWhenGeometryNotWithinTolerance() {
    Matcher<Geometry> matcher = geometryMatch(nztmHelper.box(1, 1, 10, 10), METER_TOLERANCE_NEAREST_MM);

    // geometries do not overlap at all
    assertFalse(matcher.matches(nztmHelper.box(20, 20, 30, 30)));

    assertFalse(matcher.matches(nztmHelper.box(1.0015, 1, 10, 10)));
    assertFalse(matcher.matches(nztmHelper.box(1, 1.0015, 10, 10)));
    assertFalse(matcher.matches(nztmHelper.box(1, 1, 10.0015, 10)));
    assertFalse(matcher.matches(nztmHelper.box(1, 1, 10, 10.0015)));
  }

  @Test
  public void matchesWhenGeomTopoEqual() {
    Matcher<Geometry> matcher = geometryTopoMatch(nztmHelper.line(10, 10, 20, 20));

    assertTrue(matcher.matches(nztmHelper.line(10, 10, 20, 20)));
    assertTrue(matcher.matches(nztmHelper.line(20, 20, 10, 10)));
  }

  @Test
  public void failsWhenGeomTopoNotEqual() {
    Matcher<Geometry> matcher = geometryTopoMatch(nztmHelper.line(10, 10, 20, 20));

    assertFalse(matcher.matches(nztmHelper.line(9, 9, 20, 20)));
    assertFalse(matcher.matches(nztmHelper.line(20, 20, 10.1, 10)));
  }

}
