/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.function.lang;

import static nz.org.riskscape.engine.Matchers.*;
import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.util.Arrays;
import java.util.List;

import org.junit.Before;
import org.junit.Test;

import nz.org.riskscape.engine.ArgsProblems;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.function.StringFunctions;
import nz.org.riskscape.engine.rl.BaseExpressionRealizerTest;
import nz.org.riskscape.engine.rl.DefaultOperators;
import nz.org.riskscape.engine.rl.MathsFunctions;
import nz.org.riskscape.engine.types.RSList;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.rl.ast.ExpressionProblems;

@SuppressWarnings("unchecked")
public class ZipListsTest extends BaseExpressionRealizerTest {

  ZipLists subject = new ZipLists();

  @Before
  public void addFunctions() {
    project.getFunctionSet().add(new ZipLists().identified("zip"));
    project.getFunctionSet().insertFirst(DefaultOperators.INSTANCE);
    project.getFunctionSet().addAll(MathsFunctions.FUNCTIONS);
    project.getFunctionSet().addAll(StringFunctions.FUNCTIONS);
  }

  @Test
  public void canDoSomeMathsOnElementsOfaList() throws Exception {
    evaluate("zip([1, 2, 3], [10, 20, 30], (l1, l2) -> l1 + l2)", Tuple.EMPTY_TUPLE);
    assertThat(realizationProblems, empty());
    assertThat(realized.getResultType(), equalTo(RSList.create(Types.INTEGER)));
    assertEquals(
        evaluated,
        Arrays.asList(11L, 22L, 33L)
    );
  }

  @Test
  public void extraItemsGetIgnored1() throws Exception {
    // list 1 is shortest
    evaluate("zip([1], [10, 20, 30], (l1, l2) -> l1 + l2)", Tuple.EMPTY_TUPLE);
    assertThat(realizationProblems, empty());
    assertThat(realized.getResultType(), equalTo(RSList.create(Types.INTEGER)));
    assertEquals(
        evaluated,
        Arrays.asList(11L)
    );
  }

  @Test
  public void extraItemsGetIgnored2() throws Exception {
    // list 2 is shortest
    evaluate("zip([1, 2, 3], [10, 20], (l1, l2) -> l1 + l2)", Tuple.EMPTY_TUPLE);
    assertThat(realizationProblems, empty());
    assertThat(realized.getResultType(), equalTo(RSList.create(Types.INTEGER)));
    assertEquals(
        evaluated,
        Arrays.asList(11L, 22L)
    );
  }

  @Test
  public void canPairItemsAndChangeToSimpleType() throws Exception {
    evaluate("zip([1, 2, 3], [10, 20, 30], (l1, l2) -> str(l1 + l2))", Tuple.EMPTY_TUPLE);
    assertThat(realizationProblems, empty());
    assertThat(realized.getResultType(), equalTo(RSList.create(Types.TEXT)));
    assertEquals(
        evaluated,
        Arrays.asList("11", "22", "33")
    );
  }

  @Test
  public void canPairItemsAndChangeToStructType() throws Exception {
    Struct expected = Struct.of("list1", Types.INTEGER, "list2", Types.TEXT);
    evaluate("zip([1, 2, 3], [10, 20, 30], (l1, l2) -> {list1: l1, list2: str(l2)})", Tuple.EMPTY_TUPLE);
    assertThat(realizationProblems, empty());
    assertThat(realized.getResultType(), equalTo(RSList.create(expected)));
    assertThat((List<?>) evaluated, contains(
        Tuple.ofValues(expected, 1L, "10"),
        Tuple.ofValues(expected, 2L, "20"),
        Tuple.ofValues(expected, 3L, "30")
    ));
  }

  @Test
  public void givesAnErrorIfArg1NotList() {
    evaluate("zip(1, [2], (foo, bar) -> foo + bar)", Tuple.EMPTY_TUPLE);
    assertThat(realizationProblems, contains(hasAncestorProblem(
        is(ArgsProblems.mismatch(subject.getArguments().get(0), Types.INTEGER))
    )));
  }

  @Test
  public void givesAnErrorIfArg2NotList() {
    evaluate("zip([1], '2', (foo, bar) -> foo + bar)", Tuple.EMPTY_TUPLE);
    assertThat(realizationProblems, contains(hasAncestorProblem(
        is(ArgsProblems.mismatch(subject.getArguments().get(1), Types.TEXT))
    )));
  }

  @Test
  public void givesAnErrorIfArg3NotLamba() {
    evaluate("zip([1], [2], 'foo')", Tuple.EMPTY_TUPLE);
    assertThat(realizationProblems, contains(hasAncestorProblem(
        is(ArgsProblems.mismatch(subject.getArguments().get(2), Types.TEXT))
    )));
  }

  @Test
  public void givesAnErrorIfLambaNotEnoughArgs() {
    evaluate("zip([1], [2], l -> l)", Tuple.EMPTY_TUPLE);
    assertThat(realizationProblems, contains(hasAncestorProblem(
        is(ExpressionProblems.get().lambdaArityError(parse("l -> l"), 1, 2))
    )));
  }

  @Test
  public void givesAnErrorIfLambaTooManyArgs() {
    evaluate("zip([1], [2], (l1, l2, l3) -> l1 + l2)", Tuple.EMPTY_TUPLE);
    assertThat(realizationProblems, contains(hasAncestorProblem(
        is(ExpressionProblems.get().lambdaArityError(parse("(l1, l2, l3) -> l1 + l2"), 3, 2))
    )));
  }

  @Test
  public void givesAllErrorsAtOnce() {
    evaluate("zip(1, '2', (l1, l2, l3) -> l1 + l2)", Tuple.EMPTY_TUPLE);
    assertThat(realizationProblems, hasSize(1));
    Problem found = realizationProblems.get(0);

    assertThat(found, hasAncestorProblem(
        is(ArgsProblems.mismatch(subject.getArguments().get(0), Types.INTEGER))
    ));

    assertThat(found, hasAncestorProblem(
        is(ArgsProblems.mismatch(subject.getArguments().get(1), Types.TEXT))
    ));

    assertThat(found, hasAncestorProblem(
        is(ExpressionProblems.get().lambdaArityError(parse("(l1, l2, l3) -> l1 + l2"), 3, 2))
    ));
  }

  @Test
  public void canCloseOverScopeWithoutSideEffects_GL1414() throws Exception {
    assertThat(
        evaluate("zip([1, 2], ['a', 'b'], (x, y) -> *)", Tuple.EMPTY_TUPLE),
        equalTo(tuples("[{x: 1, y: 'a'}, {x: 2, y: 'b'}]"))
    );
  }
}
