/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.function.lang;

import static nz.org.riskscape.engine.Matchers.hasAncestorProblem;
import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.equalTo;
import static org.junit.Assert.assertThat;
import java.util.List;
import org.junit.Before;
import org.junit.Test;

import nz.org.riskscape.engine.ArgsProblems;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.function.IdentifiedFunction;
import nz.org.riskscape.engine.rl.BaseExpressionRealizerTest;
import nz.org.riskscape.rl.ast.ExpressionProblems;

public class RemoveAttrTest extends BaseExpressionRealizerTest {

  @Before
  public void setup() {
    project.getFunctionSet().add(new RemoveAttr().builtin("remove_attr", IdentifiedFunction.Category.LANGUAGE));
  }

  @Test
  public void canRemoveAttributeAsStruct() {
    assertThat(
        evaluate(
            "remove_attr({first: 'first', geom: 'some-geometry', third: 'third'}, {geom: 'some-geometry'})",
            Tuple.EMPTY_TUPLE
        ),
        equalTo(evaluate("{first: 'first', third: 'third'}", Tuple.EMPTY_TUPLE))
    );
  }

  @Test
  public void canRemoveAttributeAsString() {
    assertThat(
        evaluate("remove_attr({first: 'first', geom: 'some-geometry', third: 'third'}, 'geom')", Tuple.EMPTY_TUPLE),
        equalTo(evaluate("{first: 'first', third: 'third'}", Tuple.EMPTY_TUPLE))
    );
  }

  @Test
  public void showsErrorForWrongNumberOfAttributes() {
    evaluate("remove_attr({foo: 'bar'})", Tuple.EMPTY_TUPLE);
    assertThat(realizationProblems, contains(
        hasAncestorProblem(equalTo(ArgsProblems.get().wrongNumber(2, 1)))
    ));

    evaluate("remove_attr({foo: 'bar'}, 'foo', 'bar')", Tuple.EMPTY_TUPLE);
    assertThat(realizationProblems, contains(
        hasAncestorProblem(equalTo(ArgsProblems.get().wrongNumber(2, 3)))
    ));
  }

  @Test
  public void showsErrorForRemovingNonExistingAttribute() {
    evaluate("remove_attr({foo: 'bar'}, 'baz')", Tuple.EMPTY_TUPLE);
    assertThat(realizationProblems, contains(hasAncestorProblem(
        equalTo(ExpressionProblems.get().noSuchStructMember("baz", List.of("foo")))
    )));

    evaluate("remove_attr({foo: 'bar'}, {baz: 'baz'})", Tuple.EMPTY_TUPLE);
    assertThat(realizationProblems, contains(hasAncestorProblem(
        equalTo(ExpressionProblems.get().noSuchStructMember("baz", List.of("foo")))
    )));

    // The problem is thrown here when trying to realize {foo.baz}
    evaluate("remove_attr({foo: 'bar'}, {foo.baz})", Tuple.EMPTY_TUPLE);
    assertThat(realizationProblems, contains(hasAncestorProblem(
        equalTo(ExpressionProblems.get().noSuchStructMember("foo.baz", List.of()))
    )));

  }

}
