/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.function.lang;

import static nz.org.riskscape.engine.Matchers.*;
import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import org.junit.Before;
import org.junit.Test;

import nz.org.riskscape.engine.ArgsProblems;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.function.IdentifiedFunction.Category;
import nz.org.riskscape.engine.rl.BaseExpressionRealizerTest;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.TypeProblems;
import nz.org.riskscape.engine.types.Types;

public class MergeStructTest extends BaseExpressionRealizerTest {

  @Before
  public void setup() {
    project.getFunctionSet().add(new MergeStruct().builtin("merge", Category.LANGUAGE));
  }

  @Test
  public void willNotRealizeWithoutTwoStructs() {
    evaluate("merge({})", Tuple.EMPTY_TUPLE);
    assertThat(realizationProblems, contains(
        hasAncestorProblem(equalTo(ArgsProblems.get().wrongNumber(2, 1)))
    ));

    evaluate("merge(1, {})", Tuple.EMPTY_TUPLE);
    assertThat(realizationProblems, contains(
      hasAncestorProblem(equalTo(TypeProblems.get().mismatch(parse("1"), Struct.of(), Types.INTEGER)))
    ));

    evaluate("merge({}, 'foo')", Tuple.EMPTY_TUPLE);
    assertThat(realizationProblems, contains(
      hasAncestorProblem(equalTo(TypeProblems.get().mismatch(parse("'foo'"), Struct.of(), Types.TEXT)))
    ));

    evaluate("merge(1.0, false)", Tuple.EMPTY_TUPLE);
    assertThat(realizationProblems, contains(allOf(
      hasAncestorProblem(equalTo(TypeProblems.get().mismatch(parse("1.0"), Struct.of(), Types.FLOATING))),
      hasAncestorProblem(equalTo(TypeProblems.get().mismatch(parse("false"), Struct.of(), Types.BOOLEAN)))
    )));
  }

  @Test
  public void canMergeTwoDistinctStructs() throws Exception {
    assertThat(
      evaluate("merge({foo: 1, bar: 'cool'}, {baz: true})", Tuple.EMPTY_TUPLE),
      equalTo(evaluate("{foo: 1, bar: 'cool', baz: true}", Tuple.EMPTY_TUPLE))
    );

    // check these two empty edge cases
    assertThat(
      evaluate("merge({}, {baz: true})", Tuple.EMPTY_TUPLE),
      equalTo(evaluate("{baz: true}", Tuple.EMPTY_TUPLE))
    );
    assertThat(
      evaluate("merge({foo: 1, bar: 'cool'}, {})", Tuple.EMPTY_TUPLE),
      equalTo(evaluate("{foo: 1, bar: 'cool'}", Tuple.EMPTY_TUPLE))
    );
  }

  @Test
  public void valuesFromTheRhsAreSelectedInPreferenceToLhs() throws Exception {
    assertThat(
      evaluate("merge({foo: 1, bar: 'cool'}, {foo: 2})", Tuple.EMPTY_TUPLE),
      equalTo(evaluate("{foo: 2, bar: 'cool'}", Tuple.EMPTY_TUPLE))
    );
  }

  @Test
  public void typesFromTheRhsAreSelectedInPreferenceToLhs() throws Exception {
    Object expected = evaluate("{foo: 2.0, bar: 'cool'}", Tuple.EMPTY_TUPLE);
    assertThat(
      evaluate("merge({foo: 1, bar: 'cool'}, {foo: 2.0})", Tuple.EMPTY_TUPLE),
      equalTo(expected)
    );

    assertThat(
      realized.getResultType(),
      equalTo(Struct.of("foo", Types.FLOATING, "bar", Types.TEXT))
    );
  }

  @Test
  public void aNormalizedStructIsReturned() throws Exception {
    Object expected = evaluate("{foo: 1, bar: 'cool'}", Tuple.EMPTY_TUPLE);
    assertThat(
      evaluate("merge({foo: 1}, {bar: 'cool'})", Tuple.EMPTY_TUPLE),
      equalTo(expected)
    );
    assertSame(realized.getResultType(), realizationContext.normalizeStruct(realized.getResultType().asStruct()));
  }
}

