/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.function.lang;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import org.junit.Before;
import org.junit.Test;

import nz.org.riskscape.engine.Matchers;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.function.StringFunctions;
import nz.org.riskscape.engine.rl.BaseExpressionRealizerTest;
import nz.org.riskscape.engine.rl.DefaultOperators;
import nz.org.riskscape.engine.rl.MathsFunctions;
import nz.org.riskscape.engine.types.LambdaType;
import nz.org.riskscape.engine.types.TypeProblems;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.rl.ast.ExpressionProblems;
import nz.org.riskscape.rl.ast.FunctionCall;

public class CallFunctionTest extends BaseExpressionRealizerTest {

  @Before
  public void addFunctions() {
    project.getFunctionSet().add(new Call().identified("call"));
    project.getFunctionSet().insertFirst(DefaultOperators.INSTANCE);
    project.getFunctionSet().addAll(MathsFunctions.FUNCTIONS);
    project.getFunctionSet().addAll(StringFunctions.FUNCTIONS);
  }

  @Test
  public void evaluatesTheGivenExpressionAgainstTheGivenScope() throws Exception {
    evaluate("call({foo: 1}, () -> foo + foo)", Tuple.EMPTY_TUPLE);
    assertThat(realizationProblems, empty());
    assertThat(evaluated, equalTo(2L));
    assertThat(realized.getResultType(), equalTo(Types.INTEGER));
  }

  @Test
  public void anySurroundingScopeIsIgnored() throws Exception {
    evaluate("call({foo: 1}, () -> foo + foo)", tuple("{foo: 2}"));
    assertThat(realizationProblems, empty());
    assertThat(evaluated, equalTo(2L));
    assertThat(realized.getResultType(), equalTo(Types.INTEGER));

    evaluate("call({foo: 1}, () -> foo + bar)", tuple("{bar: 2}"));
    assertThat(realizationProblems,
      contains(
        Matchers.hasAncestorProblem(
          Matchers.equalIgnoringChildren(
            ExpressionProblems.get().failedToRealize(parse("foo + bar"), tuple("{foo: 1}").getStruct())
          )
        )
      )
    );
  }

  @Test
  public void magicallyConvertsANonStructInToOne() throws Exception {
    evaluate("call(1, () -> value + value)", Tuple.EMPTY_TUPLE);
    assertThat(realizationProblems, empty());
    assertThat(evaluated, equalTo(2L));
    assertThat(realized.getResultType(), equalTo(Types.INTEGER));
  }

  @Test
  public void failsIfSecondArgNotALambda() throws Exception {
    evaluate("call(1, 2)", Tuple.EMPTY_TUPLE);

    assertThat(
        realizationProblems,
        contains(Matchers.hasAncestorProblem(
            equalTo(Problems.get(TypeProblems.class).mismatch(
                new FunctionCall.Argument(parse("2")),
                new LambdaType(),
                Types.INTEGER
            ))
        ))
    );
  }

  @Test
  public void failsIfSecondDeclaresAnyArgs() throws Exception {
    evaluate("call(1, (foo) -> foo + value)", Tuple.EMPTY_TUPLE);

    assertThat(
      realizationProblems,
      contains(Matchers.hasAncestorProblem(
          equalTo(ExpressionProblems.get().lambdaArityError(parse("(foo) -> foo + value"), 1, 0))
      ))
    );
  }



}
