/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.function.lang;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import static nz.org.riskscape.engine.Assert.*;
import static nz.org.riskscape.engine.Matchers.*;

import org.junit.Before;
import org.junit.Test;

import nz.org.riskscape.engine.ArgsProblems;
import nz.org.riskscape.engine.Matchers;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.rl.BaseExpressionRealizerTest;
import nz.org.riskscape.engine.rl.EvalException;
import nz.org.riskscape.engine.rl.LanguageFunctions;
import nz.org.riskscape.engine.rl.LogicFunctions;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;

public class AssertNotNullTest extends BaseExpressionRealizerTest {

  @Before
  public void setup() {
    project.getFunctionSet().addAll(LanguageFunctions.FUNCTIONS);
    project.getFunctionSet().addAll(LogicFunctions.LOGIC_FUNCTIONS);
  }

  @Test
  public void willStripNulls() {
    Struct inputType = Struct.of("item", Nullable.TEXT);
    assertThat(evaluate("assert_not_null(item)", tupleOfType(inputType, "{item: 'bar'}")), is("bar"));
    assertThat(realized.getResultType(), is(Types.TEXT));

    inputType = Struct.of("item", Nullable.INTEGER);
    assertThat(evaluate("assert_not_null(item)", tupleOfType(inputType, "{item: 5}")), is(5L));
    assertThat(realized.getResultType(), is(Types.INTEGER));

    Struct nested = Struct.of("foo", Nullable.TEXT);
    inputType = Struct.of("item", Nullable.of(nested));
    assertThat(
        evaluate("assert_not_null(item)", tupleOfType(inputType, "{item: {foo: 'bar'}}")),
        is(tupleOfType(nested, "{foo: 'bar'}"))
    );
    assertThat(realized.getResultType(), is(nested));
  }

  @Test
  public void doesNotRequireNullableInput() {
    // sanity check it works if the input is not nullable
    assertThat(evaluate("assert_not_null(item)", tuple("{item: 'bar'}")), is("bar"));
    assertThat(realized.getResultType(), is(Types.TEXT));

    assertThat(evaluate("assert_not_null(item)", tuple("{item: 5}")), is(5L));
    assertThat(realized.getResultType(), is(Types.INTEGER));
  }

  @Test
  public void evalFailWhenNullEncounters() {
    assertThat(
        evaluateExpectEvalException("assert_not_null(foo)", tuple("{foo: null_of('text')}")).getProblem(),
        hasAncestorProblem(is(AssertNotNull.PROBLEMS.unexpectedNull(1, parse("foo"))))
    );

    // the message can be customized
    assertThat(
        evaluateExpectEvalException("assert_not_null(foo, options: {message: 'foobar'})",
            tuple("{foo: null_of('text')}")).getProblem(),
        hasAncestorProblem(is(AssertNotNull.PROBLEMS.unexpectedNullCustomMessage("foobar")))
    );
    // an empty custom message isn't a problem itself, but it won't be used
    assertThat(
        evaluateExpectEvalException("assert_not_null(foo, options: {message: '   '})",
            tuple("{foo: null_of('text')}")).getProblem(),
        hasAncestorProblem(is(AssertNotNull.PROBLEMS.unexpectedNull(1, parse("foo"))))
    );
  }

  @Test
  public void failToRealizeWrongArgCount() {
    assertThat(
        realizeOnly("assert_not_null()", Struct.EMPTY_STRUCT),
        failedResult(Matchers.hasAncestorProblem(
            is(ArgsProblems.get().wrongNumberRange(1, 2, 0))
        ))
    );

    assertThat(
        realizeOnly("assert_not_null(3, 4, 5)", Struct.EMPTY_STRUCT),
        failedResult(Matchers.hasAncestorProblem(
            is(ArgsProblems.get().wrongNumberRange(1, 2, 3))
        ))
    );

  }

  EvalException evaluateExpectEvalException(String expr, Tuple scope) {
    return assertThrows(EvalException.class, () -> evaluate(expr, scope));
  }

}
