/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.function.benchmark;

import java.time.Duration;
import static nz.org.riskscape.engine.Assert.assertThrows;
import nz.org.riskscape.engine.RiskscapeException;
import static org.hamcrest.Matchers.allOf;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertThat;
import org.junit.Test;

public class TimerTest {

  @Test
  public void canRecordAndPrintStats() {
    Timer t = new Timer("test");
    t.start();
    t.stop();
    t.start();
    t.stop();
    assertThat(t.getStats(), allOf(
        containsString("test"),
        containsString("Round Time "),
        containsString("Mean "),
        containsString("Iterations      : 2")
    ));
  }

  @Test
  public void canResetAndCarryOn() {
    Timer t = new Timer("test");
    t.start();
    t.stop();
    t.start();
    t.reset();

    t.start();
    t.stop();
    assertThat(t.getStats(), allOf(
        containsString("Iterations      : 1")
    ));
  }

  @Test
  public void noStartsWhenNoTimeRecorded() {
    Timer t = new Timer("test");
    assertThat(t.getStats(), is("Timer 'test' has no measurements"));
  }

  @Test
  public void errorWhenStartOrStopIsBad() {
    Timer t = new Timer("test");
    RiskscapeException e = assertThrows(RiskscapeException.class, () -> t.stop());
    assertThat(e.getMessage(), is("Cannot stop timer 'test', not running"));

    t.start();
    e = assertThrows(RiskscapeException.class, () -> t.start());
    assertThat(e.getMessage(), is("Cannot start timer 'test', already running"));
  }

  @Test
  public void printsDurationToGoodScale() {
    assertThat(Timer.formatTime(Duration.ofMinutes(1)), is("  60.00 s"));
    assertThat(Timer.formatTime(Duration.ofSeconds(1)), is("   1.00 s"));
    assertThat(Timer.formatTime(Duration.ofSeconds(10).plusMillis(30)), is("  10.03 s"));

    assertThat(Timer.formatTime(Duration.ofMillis(1)), is("   1.00 ms"));
    assertThat(Timer.formatTime(Duration.ofMillis(999)), is(" 999.00 ms"));
  }

  @Test
  public void printsTimeInCorrectUnits() {
    assertThat(Timer.formatTime(1), containsString("1.00 ns"));
    assertThat(Timer.formatTime(999), containsString("999.00 ns"));
    assertThat(Timer.formatTime(1 * 1000), containsString("1.00 μs"));
    assertThat(Timer.formatTime(999 * 1000), containsString("999.00 μs"));
    assertThat(Timer.formatTime(1 * 1000 * 1000), containsString("1.00 ms"));
    assertThat(Timer.formatTime(999 * 1000 * 1000), containsString("999.00 ms"));
    assertThat(Timer.formatTime(1 * 1000 * 1000 * 1000), containsString("1.00 s"));
  }

}
