/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.expr;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;

import java.math.BigDecimal;

import org.junit.Test;

import org.locationtech.jts.geom.Geometry;

import nz.org.riskscape.engine.Assert;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.types.CoercionException;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;

public class StructAccessExpressionContextTests extends BaseStructAccessExpressionTest {

  @Test
  public void structAccessWillPluckOutSingleGeometryFieldIfAskedForAsContext() {
    Struct withGeom = Types.GEOMETRY.asStruct().parent("foo");
    Tuple tuple = new Tuple(withGeom).set("foo", new Tuple(Types.GEOMETRY.asStruct()).set("value", pointy));

    assertEquals(tuple.fetch("foo"), new StructAccessExpression("foo").evaluate(tuple));
    assertEquals(pointy, new StructAccessExpression("foo").evaluate(tuple, Geometry.class));
  }

  @Test(expected=IllegalArgumentException.class)
  public void structAccessWillFailIfNoSingleGeometryFieldWhenAskedForAsContext() {
    Struct multiGeom = Struct.of("geom1", Types.GEOMETRY).and("geom2", Types.GEOMETRY);
    Struct parentStruct = multiGeom.parent("foo");
    Tuple tuple = new Tuple(parentStruct).set("foo", new Tuple(multiGeom).set("geom1", pointy).set("geom2", pointed));

    assertEquals(tuple.fetch("foo"), new StructAccessExpression("foo").evaluate(tuple));
    new StructAccessExpression("foo").evaluate(tuple, Geometry.class);
  }

  @Test
  public void evaluateWillDelegateToRiskscapeTypeForCoercionWhenContextSupplied() throws Exception {
    Struct struct = Struct.of("int", Types.INTEGER);
    StructAccessExpression expr = expr("int");

    assertEquals("1", expr.evaluate(Tuple.ofValues(struct, 1L), String.class));
    assertEquals(Double.valueOf(5.0F), expr.evaluate(Tuple.ofValues(struct, 5L), Double.class));
    assertNull(expr.evaluate(Tuple.ofValues(struct), BigDecimal.class));
  }

  @Test
  public void evaluateWillRaiseTypeCoercionExceptionIfEvalutedValueDoesNotCoerce() throws Exception {
    Struct struct = Types.TEXT.asStruct();

    Assert.assertThrows(CoercionException.class, () -> {
      expr("value").evaluate(Tuple.ofValues(struct, "foo"), Long.class);
    });
  }

  @Test
  public void evaluateWillRaiseClassCastExceptionIfEvalutedValueDoesNotMatchContextAndNoTypeCoercionPossible()
      throws Exception {
    Struct struct = Types.TEXT.asStruct();

    Assert.assertThrows(ClassCastException.class, () -> {
      expr("value").evaluate(Tuple.ofValues(struct, "foo"), Integer.class);
    });
  }

}
