/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.data.coverage;

import static nz.org.riskscape.engine.TupleMatchers.*;
import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;
import static org.mockito.Mockito.*;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import org.junit.Before;
import org.junit.Test;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.Point;
import org.locationtech.jts.geom.Polygon;

import nz.org.riskscape.engine.ProjectTest;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.coverage.TypedCoverage;
import nz.org.riskscape.engine.function.StringFunctions;
import nz.org.riskscape.engine.function.lang.IfNull;
import nz.org.riskscape.engine.gt.NZTMGeometryHelper;
import nz.org.riskscape.engine.rl.DefaultOperators;
import nz.org.riskscape.engine.rl.MathsFunctions;
import nz.org.riskscape.engine.rl.RealizedExpression;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.engine.util.Pair;
import nz.org.riskscape.rl.ExpressionParser;
import nz.org.riskscape.rl.ast.Lambda;

public class MappedCoverageTest extends ProjectTest {

  MappedCoverage coverage;
  TypedCoverage mockCoverage;

  NZTMGeometryHelper nztmHelper = new NZTMGeometryHelper(project.getSridSet());
  Point point1 = nztmHelper.point(1, 1);
  Point point2 = nztmHelper.point(2, 2);
  Point point3 = nztmHelper.point(3, 3);
  Polygon poly1 = (Polygon) point1.buffer(2);
  Polygon poly2 = (Polygon) point2.buffer(2);
  Polygon poly3 = (Polygon) point3.buffer(2);

  @Before
  public void setup() {
    mockCoverage = mock(TypedCoverage.class, "foo");
    when(mockCoverage.getType()).thenReturn(Types.FLOATING);
    when(mockCoverage.getCoordinateReferenceSystem()).thenReturn(nztmHelper.getCrs());

    when(mockCoverage.evaluate(point1)).thenReturn(1.0D);
    when(mockCoverage.evaluate(point2)).thenReturn(2.0D);
    when(mockCoverage.evaluate(point3)).thenReturn(null);

    when(mockCoverage.getEvaluateIntersectionOp()).thenReturn(Optional.of(geom -> evaluateIntersection(geom)));

    project.getFunctionSet().insertFirst(DefaultOperators.INSTANCE);
    project.getFunctionSet().addAll(MathsFunctions.FUNCTIONS);
    project.getFunctionSet().addAll(StringFunctions.FUNCTIONS);
    project.getFunctionSet().add(new IfNull().identified("if_null"));
  }

  private List<Pair<Geometry, Object>> evaluateIntersection(Geometry geom) {
    if (geom.equals(poly1)) {
      return Arrays.asList(Pair.of(geom, 11.0D), Pair.of(geom, 12.0D));
    } else if (geom.equals(poly2)) {
      return Arrays.asList(Pair.of(geom, 200.0D));
    } else {
      return Collections.emptyList();
    }
  }

  private MappedCoverage buildCoverage(String value, String expression) {
    coverage = (MappedCoverage) MappedCoverage.build(
        realizationContext,
        mockCoverage,
        value,
        ExpressionParser.INSTANCE.parse(expression)).get();
    return coverage;
  }

  private MappedCoverage buildCoverage(String lambdaExpression) {
    coverage = (MappedCoverage) MappedCoverage.build(
        realizationContext,
        mockCoverage,
        ExpressionParser.INSTANCE.parse(lambdaExpression).isA(Lambda.class).get()).get();
    return coverage;
  }

  private MappedCoverage buildCoverage(String value, String expression, Tuple scope) {
    // rebuild scope to include the lambda args
    Struct scopeType = scope.getStruct().add(value, Types.FLOATING);
    Tuple newScope = new Tuple(scopeType);
    newScope.setAll(scope);

    RealizedExpression realized = realizationContext.getExpressionRealizer().realize(scopeType,
        ExpressionParser.INSTANCE.parse(expression)).get();
    coverage = new MappedCoverage(mockCoverage, realized, value, newScope);
    return coverage;
  }

  private List<?> sampleIntersections(Geometry geom) {
    return coverage.getEvaluateIntersectionOp().get().apply(geom)
        .stream()
        // don't care about geometry here, it's the value we're interested in
        .map(p -> p.getRight())
        .collect(Collectors.toList());
  }

  @Test
  public void canReturnValuesAsIs() {
    // this expression is kinda pointless, but proves the mocked sampling works as expected
    buildCoverage("value", "value + 1 - 1");
    assertThat(coverage.getType(), is(Types.FLOATING));
    assertThat(coverage.evaluate(point1), is(1.0D));
    assertThat(coverage.evaluate(point2), is(2.0D));
    assertNull(coverage.evaluate(point3));

    assertThat(sampleIntersections(poly1), contains(11.0D, 12.0D));
    assertThat(sampleIntersections(poly2), contains(200.0D));
    assertThat(sampleIntersections(poly3), empty());
  }

  @Test
  public void canApplySimpleTransformToValues() {
    buildCoverage("value", "value * 2");
    assertThat(coverage.getType(), is(Types.FLOATING));
    assertThat(coverage.evaluate(point1), is(2.0D));
    assertThat(coverage.evaluate(point2), is(4.0D));
    assertNull(coverage.evaluate(point3));

    assertThat(sampleIntersections(poly1), contains(22.0D, 24.0D));
    assertThat(sampleIntersections(poly2), contains(400.0D));
    assertThat(sampleIntersections(poly3), empty());
  }

  @Test
  public void canTransformTheTypeReturned() {
    buildCoverage("value", "str(value * 2)");
    assertThat(coverage.getType(), is(Types.TEXT));
    assertThat(coverage.evaluate(point1), is("2.0"));
    assertThat(coverage.evaluate(point2), is("4.0"));
    assertNull(coverage.evaluate(point3));

    assertThat(sampleIntersections(poly1), contains("22.0", "24.0"));
    assertThat(sampleIntersections(poly2), is(Arrays.asList("400.0")));
    assertThat(sampleIntersections(poly3), empty());
  }

  @Test
  public void canUseAnyVariableNameInExpression() {
    buildCoverage("x", "round(x * 2.5)");
    assertThat(coverage.getType(), is(Types.INTEGER));
    assertThat(coverage.evaluate(point1), is(3L));
    assertThat(coverage.evaluate(point2), is(5L));
    assertNull(coverage.evaluate(point3));

    assertThat(sampleIntersections(poly1), contains(28L, 30L));
    assertThat(sampleIntersections(poly2), contains(500L));
    assertThat(sampleIntersections(poly3), empty());
  }

  @Test
  public void canPreserveNullabilityOfContainedTypeInCoverage() {
    when(mockCoverage.getType()).thenReturn(Nullable.FLOATING);
    // not a null-safe expression so result should still be nullable
    buildCoverage("value", "round(value * 2)");
    assertThat(coverage.getType(), is(Nullable.INTEGER));
    assertThat(coverage.evaluate(point1), is(2L));
    assertThat(coverage.evaluate(point2), is(4L));
    assertNull(coverage.evaluate(point3));
  }

  @Test
  public void canDiscardNullabilityOfContainedTypeInCoverage() {
    when(mockCoverage.getType()).thenReturn(Nullable.FLOATING);
    // null-safe expression so the result should *never* be null
    buildCoverage("value", "str(if_null(value * 2, -1.0))");
    assertThat(coverage.getType(), is(Types.TEXT));
    assertThat(coverage.evaluate(point1), is("2.0"));
    assertThat(coverage.evaluate(point2), is("4.0"));
    assertThat(coverage.evaluate(point3), is("-1.0"));
  }

  @Test
  public void canUseInputScopeInExpression() {
    Tuple scope = Tuple.ofValues(Struct.of("foo", Types.FLOATING), 2.5D);
    buildCoverage("x", "round(x * foo)", scope);
    assertThat(coverage.getType(), is(Types.INTEGER));
    assertThat(coverage.evaluate(point1), is(3L));
    assertThat(coverage.evaluate(point2), is(5L));
    assertNull(coverage.evaluate(point3));

    assertThat(sampleIntersections(poly1), contains(28L, 30L));
    assertThat(sampleIntersections(poly2), contains(500L));
    assertThat(sampleIntersections(poly3), empty());
  }

  @Test
  public void canBuildCoverageWithLambdaExpression() {
    // the only difference here is we can infer the value name from the lambda
    buildCoverage("v -> v * 2");
    assertThat(coverage.getType(), is(Types.FLOATING));
    assertThat(coverage.evaluate(point1), is(2.0D));
    assertThat(coverage.evaluate(point2), is(4.0D));
    assertNull(coverage.evaluate(point3));

    assertThat(sampleIntersections(poly1), contains(22.0D, 24.0D));
    assertThat(sampleIntersections(poly2), contains(400.0D));
    assertThat(sampleIntersections(poly3), empty());
  }

  @Test
  public void canReturnSampledValueAsStruct() {
    buildCoverage("v -> { foo: v * 2, bar: str(v) }");
    assertThat(coverage.getType(), is(Struct.of("foo", Types.FLOATING, "bar", Types.TEXT)));
    assertThat((Tuple) coverage.evaluate(point1), tupleWithValue("foo", is(2D)));
    assertThat((Tuple) coverage.evaluate(point2), tupleWithValue("foo", is(4D)));
    assertNull(coverage.evaluate(point3));
  }

  private <T> Thread sampleRepeatedly(Point point, T expected, Consumer<T> consumeUnexpected) {
    return new Thread(() -> {
      for (int i = 0; i < 50000; i++) {
        T actual = (T) coverage.evaluate(point);
        if (!expected.equals(actual)) {
          consumeUnexpected.accept(actual);
        }
      }
    });
  }

  @Test
  public void canSampleCoverageSafelyFromMultipleThreads() {
    List<Thread> threads = new ArrayList<>();
    List<Double> unexpectedValues1 = new ArrayList<>();
    List<Double> unexpectedValues2 = new ArrayList<>();
    buildCoverage("v -> v");
    threads.add(sampleRepeatedly(point1, 1D, x -> unexpectedValues1.add(x)));
    threads.add(sampleRepeatedly(point2, 2D, x -> unexpectedValues2.add(x)));

    threads.stream().forEach(Thread::start);
    threads.stream().forEach(t -> {
      try {
        t.join();
      } catch (InterruptedException e) {
      }
    });
    assertThat(unexpectedValues1, empty());
    assertThat(unexpectedValues2, empty());
  }
}
