/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.data.coverage;

import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.Point;
import it.geosolutions.jaiext.range.NoDataContainer;
import nz.org.riskscape.engine.SRIDSet;
import nz.org.riskscape.engine.coverage.TypedCoverage;
import nz.org.riskscape.engine.gt.GeometryHelper;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.Types;
import org.geotools.coverage.grid.GridCoverage2D;
import org.geotools.geometry.Position2D;
import org.geotools.geometry.jts.ReferencedEnvelope;
import org.junit.Test;
import static org.junit.Assert.*;
import org.junit.Before;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.spy;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import org.geotools.api.coverage.PointOutsideCoverageException;
import org.geotools.api.geometry.Position;
import org.geotools.api.geometry.Bounds;
import org.geotools.api.referencing.crs.CoordinateReferenceSystem;

public class GridTypedCoverageTest {

  SRIDSet sridSet = spy(new SRIDSet());
  GeometryHelper helper = new GeometryHelper(sridSet);
  CoordinateReferenceSystem crs = helper.nzMapGrid();
  GeometryFactory factory = sridSet.getGeometryFactory(crs);

  Type type = Types.FLOATING;
  GridCoverage2D gridCoverage;
  Bounds envelope = ReferencedEnvelope.rect(0, 0, 10, 10, crs);

  TypedCoverage subject;

  Point point;

  Position directPosition = new Position2D(crs, 1, 2);

  @Before
  public void setup() {
    factory = sridSet.getGeometryFactory(crs);
    gridCoverage = mock(GridCoverage2D.class);
    when(gridCoverage.getEnvelope()).thenReturn(envelope);
    when(gridCoverage.getCoordinateReferenceSystem()).thenReturn(crs);

    subject = new GridTypedCoverage(gridCoverage, sridSet);

    point = factory.createPoint(new Coordinate(1, 2));
    when(gridCoverage.evaluate(directPosition, (double[]) null)).thenReturn(new double[] {1.8});
  }

  @Test
  public void itReturnsEmptyWhenGeometryOutsideOfGridCoverageBounds() {
    point = factory.createPoint(new Coordinate(29, 20));

    assertNull(subject.evaluate(point));
  }

  @Test
  public void itReturnsEmptyWhenGeometryIsEmpty() {
    point = factory.createPoint();

    assertNull(subject.evaluate(point));
  }

  @Test
  public void anOutsideOfCoverageExceptionReturnsNullLikeNullData() {
    when(gridCoverage.evaluate((Position) any(), (double[])any()))
      .thenThrow(new PointOutsideCoverageException());

    assertNull(subject.evaluate(point));
  }

  @Test
  public void itReturnsTheValueAtPoint() {
    assertEquals(1.8d, subject.evaluate(point));
  }

  @Test
  public void itCopesWithNaNNoDataValue() throws Exception {
    when(gridCoverage.evaluate(directPosition, (double[]) null)).thenReturn(new double[] {Double.NaN});
    NoDataContainer noDataContainer = new NoDataContainer(Double.NaN);
    when(gridCoverage.getProperty(NoDataContainer.GC_NODATA)).thenReturn(noDataContainer);
    subject = new GridTypedCoverage(gridCoverage, sridSet);
    assertNull(subject.evaluate(point));
  }

  @Test
  public void notANumberIsTheDefaultNoData() throws Exception {
    when(gridCoverage.evaluate(directPosition, (double[]) null))
        .thenReturn(new double[] {Double.NaN})
        .thenReturn(new double[] {10.0});
    // The geotools geotiff reader does not set the nodata property when the Geotiff has NaN set as nodata.
    when(gridCoverage.getProperty(NoDataContainer.GC_NODATA)).thenReturn(null);
    subject = new GridTypedCoverage(gridCoverage, sridSet);
    assertNull(subject.evaluate(point));
    assertEquals(10.0, subject.evaluate(point));
  }

  public void itReturnsTheValueAtPointInDifferenctProjection() {
    when(sridSet.reproject(point, 1)).thenReturn(point);

    point.setSRID(-99); //Manually corrupt the srid to force reprojection.
    assertEquals(1.8d, subject.evaluate(point));
    //Check the reprojection happened.
    verify(sridSet, times(1)).reproject(point, 1);
  }

}
