/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.data;

import static nz.org.riskscape.engine.Matchers.*;
import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.net.URI;
import java.nio.file.Paths;
import java.util.List;

import org.junit.Test;

import nz.org.riskscape.engine.Matchers;
import nz.org.riskscape.engine.ProjectTest;
import nz.org.riskscape.engine.coverage.TypedCoverage;
import nz.org.riskscape.engine.output.Format;
import nz.org.riskscape.engine.problem.GeneralProblems;
import nz.org.riskscape.engine.relation.EmptyRelation;
import nz.org.riskscape.engine.relation.Relation;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.TypeProblems;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problem.Severity;
import nz.org.riskscape.problem.Problems;

@SuppressWarnings("unchecked")
public class DefaultBookmarkResolversTest extends ProjectTest {

  URI bookmarkLocation = URI.create("foo://bar/baz");
  URI relativeTo = Paths.get(".").toUri();
  Bookmark bookmark = Bookmark.builder().id("foo").location(bookmarkLocation).build();
  Relation relation = new EmptyRelation(Struct.EMPTY_STRUCT);
  Problem failedWith = Problems.foundWith("uh oh");
  BookmarkResolvers resolvers = engine.getBookmarkResolvers();
  Bookmarks bookmarks = project.getBookmarks();

  @Test
  public void canResolveABookmarkJustByItsId() throws Exception {
    addPickledData("foo", relation);

    assertThat(
        resolvers.getData(bookmark, bindingContext, Relation.class),
        Matchers.result(equalTo(relation)));
  }

  @Test
  public void failedToFindAResolverGivesAProblem() throws Exception {
    assertThat(
        resolvers.getData(bookmark, bindingContext, Relation.class),
        Matchers.failedResult(hasAncestorProblem(
            is(BookmarkProblems.get().couldNotResolve("foo://bar/baz"))
        ))
    );
  }

  @Test
  public void failedToFindResolverForFormatGivesAProblem() throws Exception {
    bookmark = bookmark.withFormat("who-knows");
    assertThat(
        resolvers.getData(bookmark, bindingContext, Relation.class),
        Matchers.failedResult(hasAncestorProblem(
            is(GeneralProblems.get()
                .noSuchObjectExistsDidYouMean("who-knows", Format.class, resolvers.getSupportedFormats())
                .withChildren(BookmarkProblems.get().unknownFormatTip())
        ))
    ));
  }

  @Test
  public void failedValidationGivesAProblem() throws Exception {
    bookmarkResolver.add(ResolvedBookmark.stub(bookmark, relation, List.of(), List.of(failedWith)));

    assertThat(
        resolvers.getData(bookmark, bindingContext, Relation.class),
        Matchers.failedResult(
            allOf(
              Matchers.isProblemAffecting(Severity.ERROR, Bookmark.class),
              Matchers.hasProblems(
                  sameInstance(failedWith)
              )
            )
        )
    );
  }

  @Test
  public void failedResolvingGivesAProblem() throws Exception {
    bookmarkResolver.add(ResolvedBookmark.stub(bookmark, relation, List.of(failedWith), List.of()));

    assertThat(
        resolvers.getData(bookmark, bindingContext, Relation.class),
        Matchers.failedResult(allOf(
            Matchers.isProblemAffecting(Severity.ERROR, Bookmark.class),
            Matchers.hasProblems(
                sameInstance(failedWith)
            )
        ))
    );
  }

  @Test
  public void typeMismatchGivesAProblem() throws Exception {
    addPickledData(bookmark.getId(), relation);

    assertThat(
        resolvers.getData(bookmark, bindingContext, TypedCoverage.class),
        Matchers.failedResult(allOf(
            Matchers.isProblemAffecting(Severity.ERROR, Bookmark.class),
            Matchers.hasProblems(
                Matchers.isProblem(Severity.ERROR, TypeProblems.class, "mismatch")
            )
        ))
    );
  }
}
