/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.bind.impl;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import org.geotools.filter.text.cql2.CQLException;
import org.junit.Test;

import nz.org.riskscape.engine.Assert;
import nz.org.riskscape.engine.bind.Parameter;
import nz.org.riskscape.engine.bind.ParameterBindingException;
import nz.org.riskscape.engine.sort.SortBy;
import nz.org.riskscape.engine.sort.SortBy.Direction;
import nz.org.riskscape.rl.ast.PropertyAccess;

public class SortByBinderTest extends BaseBinderTest {

  SortByBinder subject = new SortByBinder();

  @Test
  public void itCanBindOrderBy() throws CQLException {
    Parameter param = Parameter.required("orderBy", SortBy.class);
    assertThat(subject.canBind(param), is(true));
    assertThat(subject.bind(context, param, "value"),
        equalTo(toSortBy("value", Direction.ASC)));
    assertThat(subject.bind(context, param, "value desc"),
        equalTo(toSortBy("value", Direction.DESC)));
    assertThat(subject.bind(context, param, "value DESC"),
        equalTo(toSortBy("value", Direction.DESC)));
    assertThat(subject.bind(context, param, "   value    DESC   "),
        equalTo(toSortBy("value", Direction.DESC)));
    assertThat(subject.bind(context, param, "value asc"),
        equalTo(toSortBy("value", Direction.ASC)));
    assertThat(subject.bind(context, param, "value ASC"),
        equalTo(toSortBy("value", Direction.ASC)));
    assertThat(subject.bind(context, param, "valueDESC"),
        equalTo(toSortBy("valueDESC", Direction.ASC)));
    assertThat(subject.bind(context, param, "DESC"),
        equalTo(toSortBy("DESC", Direction.ASC)));
  }

  @Test
  public void itFailsIfNotOneOrTwoWhitespaceSeparatedTokens() throws Exception {
    Assert.assertThrows(ParameterBindingException.class,
        () -> bind(SortBy.class, "foo bar desc"));

    Assert.assertThrows(ParameterBindingException.class,
        () -> bind(SortBy.class, "foo desc cool"));

    Assert.assertThrows(ParameterBindingException.class,
        () -> bind(SortBy.class, "  "));

  }
  @Test
  public void itFailsIfLastTokenIsNotAscOrDesc() throws Exception {
    Assert.assertThrows(ParameterBindingException.class,
        () -> bind(SortBy.class, "foo bar"));
    Assert.assertThrows(ParameterBindingException.class,
        () -> bind(SortBy.class, "foo des"));
  }

  private SortBy toSortBy(String path, Direction direction) {
    return new SortBy(PropertyAccess.of(path), direction);
  }

}
