/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.bind.impl;

import static nz.org.riskscape.engine.Matchers.*;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.LongAccumulator;

import org.junit.Test;
import org.mockito.Mockito;

import nz.org.riskscape.engine.bind.BindingContext;
import nz.org.riskscape.engine.problem.GeneralProblems;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.ResultOrProblems;

public class NumberBinderTest  {

  NumberBinder binder = new NumberBinder();
  BindingContext context = Mockito.mock(BindingContext.class); // not expected to be used

  @Test
  public void canBindAcceptsStringToNumber() {
    assertTrue(binder.canBind(String.class, Integer.class));
    assertTrue(binder.canBind(String.class, Long.class));
    assertTrue(binder.canBind(String.class, Float.class));
    assertTrue(binder.canBind(String.class, Double.class));
    assertTrue(binder.canBind(String.class, BigDecimal.class));
  }

  @Test
  public void canBindAcceptsNumberToNumber() {
    assertTrue(binder.canBind(Long.class, Integer.class));
    assertTrue(binder.canBind(Float.class, Long.class));
    assertTrue(binder.canBind(Double.class, Float.class));
    assertTrue(binder.canBind(BigDecimal.class, Double.class));
    assertTrue(binder.canBind(Integer.class, BigDecimal.class));
  }

  @Test
  public void canBindDoesNotAllowAllDestinationNumberTypes() {
    // some negative tests. The destination types in these tests are Number implemenations that are
    // not supported. We don't have any use for them, and if we did we'd need special code because
    // these do not have the string only constructer required by the binder.
    assertFalse(binder.canBind(String.class, AtomicInteger.class));
    assertFalse(binder.canBind(String.class, AtomicLong.class));
    assertFalse(binder.canBind(String.class, LongAccumulator.class));
  }

  @Test
  public void canTurnStringsInToVariousNumberTypes() throws Exception {
    assertBinding("1", Integer.class, 1);
    assertBinding("-1", Integer.class, -1);
    assertBinding("1", Long.class, 1L);
    assertBinding("-1", Long.class, -1L);
    assertBinding("1", Double.class, 1.0D);
    assertBinding("-1", Double.class, -1.0D);
    assertBinding("1", Float.class, 1.0F);
    assertBinding("-1", Float.class, -1.0F);
    // i doubt we use this, but here it is anyway
    assertBinding("1", BigDecimal.class, new BigDecimal("1"));
    assertBinding("-1", BigDecimal.class, new BigDecimal("-1"));

    assertBinding("1.1", Double.class, 1.1D);
    assertBinding("-1.1", Double.class, -1.1D);
    assertBinding("1.1", Float.class, 1.1F);
    assertBinding("-1.1", Float.class, -1.1F);
  }

  @Test
  public void canConvertNumberTypes() {
    assertBinding(10, Integer.class, 10);
    assertBinding(10L, Integer.class, 10);
    assertBinding(10F, Integer.class, 10);
    assertBinding(10D, Integer.class, 10);
    assertBinding(new BigDecimal("10"), Integer.class, 10);

    assertBinding(10, Long.class, 10L);
    assertBinding(10L, Long.class, 10L);
    assertBinding(10F, Long.class, 10L);
    assertBinding(10D, Long.class, 10L);
    assertBinding(new BigDecimal("10"), Long.class, 10L);

    assertBinding(10, Float.class, 10F);
    assertBinding(10L, Float.class, 10F);
    assertBinding(10F, Float.class, 10F);
    assertBinding(10D, Float.class, 10F);
    assertBinding(new BigDecimal("10"), Float.class, 10F);

    assertBinding(10, Double.class, 10D);
    assertBinding(10L, Double.class, 10D);
    assertBinding(10.5F, Double.class, 10.5D);
    assertBinding(10.5D, Double.class, 10.5D);
    assertBinding(new BigDecimal("10.8"), Double.class, 10.8D);

    assertBinding(10, BigDecimal.class, new BigDecimal("10"));
    assertBinding(10L, BigDecimal.class, new BigDecimal("10"));
    assertBinding(10.5F, BigDecimal.class, new BigDecimal("10.5"));
    assertBinding(10.5D, BigDecimal.class, new BigDecimal("10.5"));
    assertBinding(new BigDecimal("10.8"), BigDecimal.class, new BigDecimal("10.8"));
  }

  @Test
  public void noWarningIfNoPrecisionLoss() {
    assertBindingWithNoPrecisionLoss(10L, Integer.class, 10);
    assertBindingWithNoPrecisionLoss(10F, Integer.class, 10);
    assertBindingWithNoPrecisionLoss(10D, Integer.class, 10);
    assertBindingWithNoPrecisionLoss(new BigDecimal(10), Integer.class, 10);

    assertBindingWithNoPrecisionLoss(10F, Long.class, 10L);
    assertBindingWithNoPrecisionLoss(10D, Float.class, 10F);
    assertBindingWithNoPrecisionLoss(new BigDecimal("10"), Double.class, 10D);
  }

  @Test
  public void precisionLossIsWarned() {
    assertBindingWithPrecisionLoss(10.5D, Integer.class, 10);
    assertBindingWithPrecisionLoss(10.5D, Long.class, 10L);
  }


  @Test
  public void aNumberFormatExceptionGetsTurnedInToAniceErrorMessage() throws Exception {
    assertThat(
      binder.bindValue(context, "1.2", Integer.class),
      failedResult(equalTo(NumberBinder.PROBLEMS.numberFormatException("1.2", Integer.class)))
    );
  }

  private <T> void assertBinding(Object bindValue, Class<T> targetClass, T expectedResult) {
    assertThat(binder.bindValue(context, bindValue, targetClass), result(expectedResult));
  }

  private <T> void assertBindingWithNoPrecisionLoss(Number input, Class<T> destinationType, Number expected) {
    List<Problem> warnings = new ArrayList<>();
    ResultOrProblems<T> result = binder.bindValue(context, input, destinationType).drainWarnings(p -> warnings.add(p));
    assertThat(result, result(expected));
    assertThat(warnings, hasSize(0));
  }

  private <T> void assertBindingWithPrecisionLoss(Number input, Class<T> destinationType, Number expected) {
    List<Problem> warnings = new ArrayList<>();
    ResultOrProblems<T> result = binder.bindValue(context, input, destinationType).drainWarnings(p -> warnings.add(p));
    assertThat(result, result(expected));
    assertThat(warnings, contains(GeneralProblems.get().precisionLoss(input, expected)));
  }

}
