/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.bind.impl;


import static nz.org.riskscape.engine.Assert.*;
import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import org.junit.Before;
import org.junit.Test;

import nz.org.riskscape.engine.bind.Parameter;
import nz.org.riskscape.engine.bind.ParameterBindingException;
import nz.org.riskscape.engine.function.ExpressionAggregationFunction;
import nz.org.riskscape.engine.function.IdentifiedFunction;
import nz.org.riskscape.engine.function.IdentifiedFunction.Category;
import nz.org.riskscape.engine.rl.FunctionCallPrototype;
import nz.org.riskscape.engine.rl.agg.AggregationFunction;

public class FunctionCallPrototypeBinderTest extends BaseBinderTest {

  static final Parameter PROTOTYPE = Parameter.required("fc", FunctionCallPrototype.class);

  FunctionCallPrototypeBinder binder = new FunctionCallPrototypeBinder();

  // dummy agg function for testing
  public final IdentifiedFunction aggFunction =
      AggregationFunction.asFunction(ExpressionAggregationFunction.builder()
          .map("value -> value")
          .reduce("(lhs, rhs) -> lhs")
          .build())
      .builtin("agg", Category.LANGUAGE);

  @Before
  public void addAggFunction() {
    project.getFunctionSet().add(aggFunction);
  }

  @Test
  public void canBind() {
    assertThat(binder.canBind(String.class, FunctionCallPrototype.class), is(true));
  }

  @Test
  public void bindsByIdOnly() {
    assertThat(bind(FunctionCallPrototype.class, "agg").getPrototype().toSource(),
        is("agg()"));
  }

  @Test
  public void bindsWithSuppliedArgs() {
    assertThat(bind(FunctionCallPrototype.class, "agg()").getPrototype().toSource(),
        is("agg()"));

    assertThat(bind(FunctionCallPrototype.class, "agg(a1)").getPrototype().toSource(),
        is("agg(a1)"));

    assertThat(bind(FunctionCallPrototype.class, "agg(a1, a2, a3)").getPrototype().toSource(),
        is("agg(a1, a2, a3)"));
  }

  @Test
  public void bindFailsWhen() {
    // a number cannot be a function ID
    assertThrows(ParameterBindingException.class,
        () -> bind(FunctionCallPrototype.class, "7"));

    // a property access with more then one identifier can't be a function id
    assertThrows(ParameterBindingException.class,
        () -> bind(FunctionCallPrototype.class, "agg.bar"));
    // nor can these
    assertThrows(ParameterBindingException.class,
        () -> bind(FunctionCallPrototype.class, "agg.bar()"));
    assertThrows(ParameterBindingException.class,
        () -> bind(FunctionCallPrototype.class, "agg.bar(a1)"));

    // the given value is not a valid function
    assertThrows(ParameterBindingException.class,
        () -> bind(FunctionCallPrototype.class, "foo"));
  }

}
