/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.bind.impl;

import static nz.org.riskscape.engine.Matchers.*;
import static org.hamcrest.Matchers.*;
import static org.hamcrest.Matchers.contains;
import static org.junit.Assert.*;
import static org.mockito.ArgumentMatchers.*;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.*;

import java.io.IOException;
import java.net.URISyntaxException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.mockito.ArgumentCaptor;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Sets;

import nz.org.riskscape.engine.Assert;
import nz.org.riskscape.engine.Engine;
import nz.org.riskscape.engine.Matchers;
import nz.org.riskscape.engine.ProjectTest;
import nz.org.riskscape.engine.TemporaryDirectoryTestHelper;
import nz.org.riskscape.engine.bind.Parameter;
import nz.org.riskscape.engine.bind.ParameterBindingException;
import nz.org.riskscape.engine.bind.ParameterField;
import nz.org.riskscape.engine.coverage.TypedCoverage;
import nz.org.riskscape.engine.data.BaseBookmarkResolver;
import nz.org.riskscape.engine.data.Bookmark;
import nz.org.riskscape.engine.data.BookmarkParameters;
import nz.org.riskscape.engine.data.BookmarkProblems;
import nz.org.riskscape.engine.data.BookmarkResolver;
import nz.org.riskscape.engine.data.ResolvedBookmark;
import nz.org.riskscape.engine.function.ArgumentList;
import nz.org.riskscape.engine.function.BaseRealizableFunction;
import nz.org.riskscape.engine.function.IdentifiedFunction.Category;
import nz.org.riskscape.engine.function.RiskscapeFunction;
import nz.org.riskscape.engine.relation.EmptyRelation;
import nz.org.riskscape.engine.relation.Relation;
import nz.org.riskscape.engine.rl.RealizationContext;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.TypeProblems;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.engine.types.WithMetadata;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problem.Severity;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ast.FunctionCall;

public class BookmarkBinderTest extends ProjectTest implements TemporaryDirectoryTestHelper {

  // we use bookmark_test() to check we can bind against expression, if that
  // expression yields a bookmark in WithMetadata that it returns
  public static class BookmarkTestFunction extends BaseRealizableFunction {
    public BookmarkTestFunction() {
      super(ArgumentList.create("id", Types.TEXT), Types.ANYTHING);
    }

    @Override
    public ResultOrProblems<RiskscapeFunction> realize(RealizationContext context, FunctionCall functionCall,
        List<Type> argumentTypes) {
      // just lookup the bookmark that matches the ID, if any
      Bookmark bookmark = arguments.getRequiredArgument(functionCall, "id")
        .flatMap(idArg -> idArg.evaluateConstant(context, String.class, Types.TEXT))
        .flatMap(id -> context.getProject().getBookmarks().getOr(id))
        .orElse(null);
      return ResultOrProblems.of(RiskscapeFunction.create(this, argumentTypes,
          bookmark == null ? Types.TEXT : WithMetadata.wrap(Types.TEXT, bookmark),
          args -> "foo"));
    }
  }

  Parameter relationParameter = Parameter.optional("relation", Relation.class);
  Parameter coverageParameter = Parameter.optional("coverage", TypedCoverage.class);
  Parameter resolvedParameter = Parameter.optional("resolved", ResolvedBookmark.class);

  BookmarkBinder binder = new BookmarkBinder();
  Path tmpdir;

  TypedCoverage coverage = TypedCoverage.empty(Types.INTEGER);
  ResolvedBookmark resolvedCoverage;
  Relation relation = new EmptyRelation(Types.TEXT.asStruct());
  ResolvedBookmark resolvedRelation;
  BookmarkResolver mockResolver = mock(BookmarkResolver.class);
  ArgumentCaptor<Bookmark> captor = ArgumentCaptor.forClass(Bookmark.class);

  Problem problem = Problems.foundWith("Test");

  ParameterBindingException ex;

  @Before
  public void setup() throws IOException, URISyntaxException {
    this.tmpdir = createTempDirectory("binder");
    resolvedCoverage = addPickledData("coverage", coverage);
    resolvedRelation = addPickledData("relation", relation);

    when(mockResolver.getId()).thenReturn("mock-resolver");
    engine.getBookmarkResolvers().add(mockResolver);
    project.getFunctionSet().add(new BookmarkTestFunction().builtin("bookmark_test", Category.MISC));
  }

  @After
  public void teardown() throws Exception {
    this.remove(tmpdir);
  }

  @Test
  public void willBindCoverageFromBookmarkIfIdMatchesValue() {
    assertSame(coverage, binder.bind(bindingContext, coverageParameter, "coverage"));
  }

  @Test
  public void willBindRelationFromBookmarkIfIdMatchesValue() {
    assertSame(relation, binder.bind(bindingContext, relationParameter, "relation"));
  }

  @Test
  public void willBindQuotedBookmarkIdentifier() {
    // single quotes are stripped
    assertSame(relation, binder.bind(bindingContext, relationParameter, "'relation'"));
    // so are double quotes
    assertSame(relation, binder.bind(bindingContext, relationParameter, "\"relation\""));
  }

  @Test
  public void willBindRelationFromBookmarkIfCoverageIsConvertible() {
    TypedCoverage convertibleCoverage = new TypedCoverage.Empty(Types.INTEGER) {
      @Override
      public Optional<Relation> asRelation() {
        return Optional.of(relation);
      }
    };
    addPickledData("convertible-coverage", convertibleCoverage);

    assertSame(relation, binder.bind(bindingContext, relationParameter, "convertible-coverage"));
  }
//
  @Test
  public void willBindResolvedBookmarkIfIdMatchesValue() {
    assertSame(resolvedCoverage, binder.bind(bindingContext, resolvedParameter, "coverage"));
  }

  @Test
  public void willThrowIfTypeMismatch() throws Exception {
    Bookmark bookmark = resolvedRelation.getBookmark();

    ex = Assert.assertThrows(ParameterBindingException.class,
        () -> binder.bind(bindingContext, coverageParameter, "relation"));

    assertThat(
      ex.getProblem(),
      equalTo(
        Problems.foundWith(
          coverageParameter,
            TypeProblems.get().mismatch(bookmark, TypedCoverage.class, EmptyRelation.class))
      )
    );
  }

  @Test
  public void willIncludeAWarningIfWeFailedToConvertCoverageToARelation() throws Exception {
    Bookmark bookmark = resolvedCoverage.getBookmark();

    ex = Assert.assertThrows(ParameterBindingException.class,
        () -> binder.bind(bindingContext, relationParameter, "coverage"));

    // we give the user a hint that we tried to convert it, but it didn't work
    assertThat(
      ex.getProblem(),
      equalTo(
        Problems.foundWith(
          relationParameter,
          TypeProblems.get().mismatch(bookmark, Relation.class, TypedCoverage.Empty.class),
          BookmarkBinder.PROBLEMS.couldNotConvert()
        )
      )
    );
  }

  @Test
  public void willCreateABookmarkOnTheFlyIfNoneRegistered() throws Exception {
    when(mockResolver.resolve(captor.capture(), same(bindingContext))).thenReturn(Optional.of(resolvedCoverage));
//
    assertSame(resolvedCoverage, binder.bind(bindingContext, resolvedParameter, "unknown-bookmark"));
    Bookmark created = captor.getValue();
    assertNotNull(created);
    assertTrue(created.getLocation().getPath().endsWith("unknown-bookmark"));
  }

  @Test
  public void willCreateABookmarkWithQueryParametersAsBookmarkOptionsIfNoneRegistered() throws Exception {
    when(mockResolver.resolve(captor.capture(), same(bindingContext))).thenReturn(Optional.of(resolvedCoverage));
    assertSame(resolvedCoverage, binder.bind(bindingContext, resolvedParameter, "this-might-be-a-file.gdb?layer=foo"));

    Bookmark created = captor.getValue();
    assertNotNull(created);
    assertTrue(created.getLocation().toString().endsWith("this-might-be-a-file.gdb?layer=foo"));
    assertEquals("layer=foo", created.getLocation().getQuery());
    assertEquals(Arrays.asList("foo"), created.getUnparsed().get("layer"));
  }

  @Test
  public void willThrowIfBookmarkHasValidationProblems() throws Exception {
    addPickledBookmark(ResolvedBookmark.withId("problematic", relation, List.of(), List.of(problem)));

    ex = Assert.assertThrows(ParameterBindingException.class,
        () -> binder.bind(bindingContext, relationParameter, "problematic"));

    assertThat(
        ex.getProblem(),
        allOf(
            Matchers.isProblemAffecting(Severity.ERROR, Parameter.class),
            Matchers.hasProblems(
                allOf(
                    Matchers.isProblemAffecting(Severity.ERROR, Bookmark.class),
                    Matchers.hasProblems(
                        is(problem)
                    )
                )
            )
        )
    );
  }

  @Test
  public void willThrowIfBookmarkHasDataProblems() throws Exception {
    ResolvedBookmark validationProblems =
        ResolvedBookmark.withId("problematic", relation, List.of(problem), List.of());

    addPickledBookmark(validationProblems);

    ex = Assert.assertThrows(ParameterBindingException.class,
        () -> binder.bind(bindingContext, relationParameter, "problematic"));

    assertThat(
        ex.getProblem(),
        allOf(
            Matchers.isProblemAffecting(Severity.ERROR, Parameter.class),
            Matchers.hasProblems(
                allOf(
                    Matchers.isProblemAffecting(Severity.ERROR, Bookmark.class),
                    Matchers.hasProblems(
                        is(problem)
                    )
                )
            )
        )
    );
  }

  @Test
  public void willThrowIfBookmarkHasMismatchingCoverageDataProblems() throws Exception {
    // covers the case where we want a relation, but it's a coverage - we want to show the original errors if they are
    // there
    ResolvedBookmark validationProblems =
        ResolvedBookmark.withId("problematic", coverage, List.of(problem), List.of());

    addPickledBookmark(validationProblems);

    ex = Assert.assertThrows(ParameterBindingException.class,
        () -> binder.bind(bindingContext, relationParameter, "problematic"));

    assertThat(
        ex.getProblem(),
        allOf(
            Matchers.isProblemAffecting(Severity.ERROR, Parameter.class),
            Matchers.hasProblems(
                allOf(
                    Matchers.isProblemAffecting(Severity.ERROR, Bookmark.class),
                    Matchers.hasProblems(
                        is(problem)
                    )
                )
            )
        )
    );
  }

  @Test
  public void willThrowIfBookmarkHasCoverageDataProblems() throws Exception {
    // covers the case where we want a relation, but it's a coverage - we want to show the original errors if they are
    // there
    ResolvedBookmark validationProblems =
        ResolvedBookmark.withId("problematic", coverage, List.of(problem), List.of());

    addPickledBookmark(validationProblems);

    ex = Assert.assertThrows(ParameterBindingException.class,
        () -> binder.bind(bindingContext, coverageParameter, "problematic"));

    assertThat(
        ex.getProblem(),
        allOf(
            Matchers.isProblemAffecting(Severity.ERROR, Parameter.class),
            Matchers.hasProblems(
                allOf(
                    Matchers.isProblemAffecting(Severity.ERROR, Bookmark.class),
                    Matchers.hasProblems(
                        is(problem)
                    )
                )
            )
        )
    );
  }

  @Test
  public void warningsProducedConsistentlyRegardlessOfParameterType() throws Exception {
    // mock a bookmark that only returns a warning when we try to get the data
    problem = problem.withSeverity(Severity.WARNING);
    ResolvedBookmark mockBookmark = mock(ResolvedBookmark.class);
    when(mockBookmark.getId()).thenReturn("has-warnings");
    when(mockBookmark.getDataType()).thenAnswer(i -> Relation.class);
    when(mockBookmark.getData(any())).then(i -> ((Class<?>) i.getArgument(0)).isAssignableFrom(Relation.class) ?
        ResultOrProblems.of(mock(Relation.class), problem) : ResultOrProblems.error("bad type"));
    bookmarkResolver.add(mockBookmark);

    // should return warnings consistently, regardless of what type we're binding against
    assertThat(binder.bindValue(bindingContext, "has-warnings", Relation.class).getProblems(),
        contains(problem));
    assertThat(binder.bindValue(bindingContext, "has-warnings", ResolvedBookmark.class).getProblems(),
        contains(problem));
  }

  @Test
  public void willThrowIfBookmarkHasDataProblemsEvenIfParameterWantsResolvedBookmark() throws Exception {
    addPickledBookmark(
        ResolvedBookmark.withId("problematic", relation, Arrays.asList(problem), Collections.emptyList()));

    ex = Assert.assertThrows(ParameterBindingException.class,
        () -> binder.bind(bindingContext, resolvedParameter, "problematic"));

    assertThat(
        ex.getProblem(),
        allOf(
            Matchers.isProblemAffecting(Severity.ERROR, Parameter.class),
            Matchers.hasProblems(
                allOf(
                    Matchers.isProblemAffecting(Severity.ERROR, Bookmark.class),
                    Matchers.hasProblems(
                        is(problem)
                    )
                )
            )
        )
    );
  }

  public static class Params extends BookmarkParameters {

    @ParameterField
    public String important;

    @Override
    public Class<?> getDataType() {
      return Params.class;
    }

  }

  public static class TestResolver extends BaseBookmarkResolver<Params> {

    public TestResolver(Engine engine) {
      super(engine);
    }

    @Override
    protected Map<String, String> getExtensionsToFormats() {
      return ImmutableMap.of("ext", "testy");
    }

    @Override
    protected ResultOrProblems<?> build(Params parameters) {
      return ResultOrProblems.of(parameters);
    }

  }

  @Test
  public void fileIsPopulatedEvenWhenUriHasQueryParams() throws Exception {
    engine.getBookmarkResolvers().add(new TestResolver(engine));
    project.getTypeSet().add("a-type", Struct.of());
    Path csvFile = Paths.get("src", "test", "resources", "nz", "org", "riskscape", "engine", "data", "dummy.ext");
    String pathWithParams = csvFile.toString() + "?important=foo&extra=bar";

    ResultOrProblems<ResolvedBookmark> resolved = binder.bindValue(project.newBindingContext(), pathWithParams,
        ResolvedBookmark.class);
    assertFalse(resolved.hasErrors());
    List<Problem> warnings = new ArrayList<>();
    Params params = resolved
        .drainWarnings(warning -> warnings.add(warning))
        .get().getData(Params.class).get();

    // we expect a warning about the unexpected param (extra) that was extracted from the query. `important` is
    // a valid parameter for the test resolver
    assertThat(warnings, contains(
        Problem.warning("Unrecognized parameters given - %s - these will be ignored", Sets.newHashSet("extra"))
    ));

    assertTrue(params.location.toString().endsWith(pathWithParams));
    assertEquals("foo", params.important);
  }

  @Test
  public void willBindBookmarkFromValidExpression() {
    assertSame(relation, binder.bind(bindingContext, relationParameter, "bookmark_test('relation')"));
    assertSame(coverage, binder.bind(bindingContext, coverageParameter, "bookmark_test('coverage')"));
    assertSame(resolvedRelation, binder.bind(bindingContext, resolvedParameter, "bookmark_test('relation')"));
    assertSame(resolvedCoverage, binder.bind(bindingContext, resolvedParameter, "bookmark_test('coverage')"));
  }

  @Test
  public void willNotBindBookmarkFromInvalidExpression() {
    // valid bookmark ID, invalid expression - there's no closing ')'
    ex = Assert.assertThrows(ParameterBindingException.class,
        () -> binder.bind(bindingContext, relationParameter, "bookmark_test('relation'"));

    assertThat(ex.getProblem(),
        hasAncestorProblem(is(BookmarkProblems.get().notBookmarkOrFile("bookmark_test('relation'"))));
  }

  @Test
  public void willNotBindBookmarkFromBadBookmarkId() {
    // valid expression, but bad bookmark ID
    ex = Assert.assertThrows(ParameterBindingException.class,
        () -> binder.bind(bindingContext, coverageParameter, "bookmark_test('unknown relation')"));

    assertThat(ex.getProblem(),
        hasAncestorProblem(is(BookmarkProblems.get().notBookmarkOrFile("bookmark_test('unknown relation')"))));

    ex = Assert.assertThrows(ParameterBindingException.class,
        () -> binder.bind(bindingContext, coverageParameter, "'foo'"));
    assertThat(ex.getProblem(),
        hasAncestorProblem(is(BookmarkProblems.get().notBookmarkOrFile("foo"))));
  }

  @Test
  public void willCreateABookmarkFromExpressionOnTheFlyIfNoneRegistered() throws Exception {
    // use mocking to mimic turning a URI into a bookmark
    when(mockResolver.resolve(captor.capture(), same(bindingContext))).thenReturn(Optional.of(resolvedCoverage));

    assertSame(resolvedCoverage, binder.bind(bindingContext, resolvedParameter, "bookmark_test('foo.shp')"));
    Bookmark created = captor.getValue();
    assertNotNull(created);
    assertTrue(created.getLocation().getPath().endsWith("bookmark_test('foo.shp')"));
  }

}
