/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.bind;


import static org.junit.Assert.*;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.BiFunction;

import org.junit.Test;

import nz.org.riskscape.engine.Assert;

public class ParameterTest {

  @Test
  public void withNewDefaultsSetNewDefaultsOnParameterWithExistingDefaults() {
    assertEquals(
        Arrays.asList("foo", "bar", "baz"),
        Parameter.required("param", String.class, "the default")
          .withNewDefaults((bc, p) ->
            Arrays.asList("foo", "bar", "baz"))
        .getDefaultValues(null));
  }

  @Test
  public void withNewDefaultsSetsDefaultsOnParameterWithNoExistingDefaults() {
    assertEquals(
        Arrays.asList("cool", "story"),
        Parameter.required("param", String.class)
          .withNewDefaults((bc, p) -> Arrays.asList("cool", "story")).getDefaultValues(null));
  }

  @Test
  public void canNotConstructAParameterWithMinGreaterThanMax() throws Exception {
    Assert.assertThrows(IllegalArgumentException.class, () -> Parameter.range("foo", String.class, 1, 0));
  }

  @Test
  public void canCloneAParameterWithANewName() throws Exception {
    Parameter toClone = new Parameter(
        "name",
        String.class,
        Optional.of((b, p) -> Collections.emptyList()),
        3,
        4);

    Parameter cloned = toClone.withNewName("my new name");

    assertNotEquals(toClone, cloned);
    assertEquals(toClone, cloned.withNewName("name"));

    assertEquals("my new name", cloned.getName());
    assertEquals(toClone.getDefaultFunction(), cloned.getDefaultFunction());
    assertEquals(toClone.getMaxRequired(), cloned.getMaxRequired());
    assertEquals(toClone.getMinRequired(), cloned.getMinRequired());
    assertEquals(toClone.getType(), cloned.getType());
  }

  @Test
  public void getArityTests() throws Exception {
    assertEquals("0..1", Parameter.optional("optional", String.class).getArity());
    assertEquals("1", Parameter.required("required", String.class).getArity());
    assertEquals("5..100", Parameter.range("foo", String.class, 5, 100).getArity());
    assertEquals("0+", Parameter.range("foo", String.class, 0, Integer.MAX_VALUE).getArity());
    assertEquals("1+", Parameter.range("foo", String.class, 1, Integer.MAX_VALUE).getArity());
  }

  @Test(expected=IllegalArgumentException.class)
  public void canNotCreateParametersWithNegativeArity() throws Exception {
    Parameter.range("foo", String.class, -1, 1);
  }

  @Test(expected=IllegalArgumentException.class)
  public void canNotCreateParametersWithMaxZero() throws Exception {
    Parameter.range("foo", String.class, 0, 0);
  }

  @Test
  public void canCloneWithANewType() throws Exception {
    BiFunction<BindingContext, Parameter, List<?>> func = (a, b) -> Collections.singletonList("cool");
    Parameter startWith = Parameter.required("foo", String.class).withNewDefaults(func);
    Parameter cloned = startWith.withNewType(Long.class);

    assertSame(startWith.getName(), cloned.getName());
    assertSame(startWith.getMaxRequired(), cloned.getMaxRequired());
    assertSame(startWith.getMinRequired(), cloned.getMinRequired());
    assertSame(startWith.getDefaultFunction(), cloned.getDefaultFunction());

    assertNotSame(startWith.getType(), cloned.getType());
    assertSame(Long.class, cloned.getType());
  }

}
