/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.bind;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.util.Arrays;

import org.junit.Test;

import lombok.RequiredArgsConstructor;
import nz.org.riskscape.engine.Assert;
import nz.org.riskscape.engine.Matchers;
import nz.org.riskscape.engine.ProjectTest;
import nz.org.riskscape.engine.bind.impl.EnumBinder;
import nz.org.riskscape.engine.bind.impl.SortByBinder;
import nz.org.riskscape.engine.problem.GeneralProblems;
import nz.org.riskscape.engine.sort.SortBy;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.Problem.Severity;
import nz.org.riskscape.rl.ExpressionParser;

/**
 * Test some bits of pieces of the half-way refactored code
 */
public class BindingCompatibilityTest extends ProjectTest {

  SortByBinder binder  = new SortByBinder();
  EnumBinder enumBinder = new EnumBinder();

  @RequiredArgsConstructor
  public class MockInput {

    private final String value;
    @Override
    public String toString() {
      return value;
    }
  }

  @Test
  public void oldStyleBindersAcceptAnyObjectByToStringingThem() throws Exception {
    assertTrue(binder.canBind(Object.class, SortBy.class));
    assertThat(
      binder.bindValue(bindingContext, new MockInput("foo"), SortBy.class),
      Matchers.result(equalTo(new SortBy(ExpressionParser.parseString("foo"), SortBy.Direction.ASC)))
    );
  }


  @Test
  public void oldStyleBindersProduceDecentErrorMessages() throws Exception {
    assertThat(
      binder.bindValue(bindingContext, new MockInput("test around"), Long.class),
      Matchers.failedResult(equalTo(
          new Problem(Severity.ERROR, "Bad direction 'around', should be one of [ASC, DESC]")
      ))
    );

    Parameter parameter = Parameter.required("foo", SortBy.class);

    assertThat(
        bindingContext.bind("test around", parameter),
      Matchers.failedResult(equalTo(
          ParamProblems.get().bindingError(parameter,
              new Problem(Severity.ERROR, "Bad direction 'around', should be one of [ASC, DESC]")
          )
      ))
    );
  }

  public enum Values {
    FOO,
    BAR,
    BAZ
  }

  @Test
  public void newStyleBindersStillWorkWithTheOldBindingAPI() throws Exception {
    Parameter valueParam = Parameter.required("value", Values.class);
    assertEquals(Values.FOO, enumBinder.bind(bindingContext, valueParam, "foo"));

    ParameterBindingException ex =
      Assert.assertThrows(ParameterBindingException.class, () -> enumBinder.bind(bindingContext, valueParam, "far"));

    assertEquals(valueParam, ex.getModelParameter());
    assertEquals(
      Problems.foundWith(
        valueParam,
        GeneralProblems.get().notAnOption("far", Values.class, Arrays.asList(Values.values()))
      ),
      ex.getProblem()
    );

    // we get the same error, regardless of how the old api is called
    ParameterBindingException ex2 =
        Assert.assertThrows(ParameterBindingException.class, () ->  bindingContext.bind(valueParam, "far"));

    assertEquals(ex.getProblem(), ex2.getProblem());
  }

}
