/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine;

import java.util.HashMap;

import com.google.common.collect.Maps;

import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;

public class TupleSpeedTest {

  private Struct struct;

  protected Tuple constructStructMap(String[] keys, Object[] values) {
    Tuple structMap = new Tuple(struct);
    for (int i = 0; i < keys.length; i++) {
      structMap.set(keys[i], values[i]);
    }

    return structMap;
  }

  protected HashMap<String, Object> constructHashMap(String[] keys, Object[] values) {
    HashMap<String, Object> hashMap = Maps.newHashMapWithExpectedSize(keys.length);
    for (int i = 0; i < keys.length; i++) {
      hashMap.put(keys[i], values[i]);
    }

    return hashMap;
  }


  public static void main(String[] args) {
    TupleSpeedTest structMapSpeedTest = new TupleSpeedTest();
    structMapSpeedTest.runIt(10000000);
  }

  private void runIt(int loops) {
    // blind/vain attempt to make sure jvm doesn't optimise by not actually calling the function
    String[] keys = new String[] {"foo", "bar", "baz", "pete", "tong", "smelly", "britches", "cat", "dog", "mouse",
        "cow"};
    Struct.StructBuilder builder = new Struct.StructBuilder();
    Object[] values = new Object[keys.length];
    for (int i = 0; i < keys.length; i++) {
      builder.and(keys[i], Types.ANYTHING);
      values[i] = new Object();

    }
    struct = builder.build();

    long time;

    for (int iterations = 0; iterations < 5; iterations++) {

      time = timeIt(loops, () -> constructHashMap(keys, values));
      System.out.println(String.format("construct hashmap took %d", time));

      time = timeIt(loops, () -> constructStructMap(keys, values));
      System.out.println(String.format("construct structmap took %d", time));

      Tuple structMap = constructStructMap(keys, values);
      time = timeIt(loops, () -> accessStructMap(structMap, keys));
      System.out.println(String.format("access structmap took %d", time));

      HashMap<String, Object> hashMap = constructHashMap(keys, values);
      time = timeIt(loops, () -> accessHashMap(hashMap, keys));
      System.out.println(String.format("access hashMap took %d", time));
    }

  }

  private void accessStructMap(Tuple map, String[] keys) {
    for (String key : keys) {
      map.fetch(key);
    }
  }

  private void accessHashMap(HashMap<String, Object> map, String[] keys) {
    for (String key : keys) {
      map.get(key);
    }
  }


  private long timeIt(int loops, Runnable method) {
    long time = System.currentTimeMillis();
    for (int i = 0; i < loops; i++) {
      method.run();
    }
    return System.currentTimeMillis() - time;
  }
}
