/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine;

import org.junit.Before;

import org.junit.Test;

import com.google.common.collect.ImmutableMap;

import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Value;
import nz.org.riskscape.engine.typeset.IdentifiedType;

import static org.junit.Assert.*;

import java.math.BigDecimal;
import java.util.Map;

public class JsonHelperTest {

  private JsonHelper subject;

  @Before
  public void setup() {
    subject = new JsonHelper();
  }

  @Test
  public void canParseAString() {
    Value anything = subject.deserializeValue(Types.ANYTHING, "hi there");
    assertEquals("hi there", anything.getValue());
    Value text = subject.deserializeValue(Types.TEXT, "hi there");
    assertEquals("hi there", text.getValue());
  }

  @Test
  public void canParseAnInteger() {
    Value anything = subject.deserializeValue(Types.ANYTHING, "1");
    assertEquals(1, anything.getValue());
    Value anint = subject.deserializeValue(Types.INTEGER, "-99");
    assertEquals(-99L, anint.getValue());
  }

  @Test
  public void canParseADecimal() {
    Value anything = subject.deserializeValue(Types.ANYTHING, "1.1235");
    assertEquals(1.1235, anything.getValue());
    Value decimal = subject.deserializeValue(Types.DECIMAL, "1.1235");
    assertEquals(new BigDecimal(1.1235), decimal.getValue());
  }

  @Test
  public void canParseAStruct() {
    String json = "{\"foo\":\"bar\"}";
    Map<String, String> map = ImmutableMap.<String, String>builder().put("foo", "bar").build();

    Value anything = subject.deserializeValue(Types.ANYTHING, json);
    assertEquals(map, anything.getValue());

    Struct struct = Struct.of("foo", Types.ANYTHING).build();
    Value value = subject.deserializeValue(struct, json);
    assertEquals(map, ((Tuple) value.getValue()).toMap());
  }

  @Test
  public void canParseAValue() {
    IdentifiedType type = Helpers.TEXT;
    Value asset = subject.deserializeValue(type, "wooden building");
    assertEquals("wooden building", asset.getValue());
  }


}
