/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine;

import static org.hamcrest.Matchers.*;
import static org.hamcrest.Matchers.contains;
import static org.junit.Assert.*;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.*;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;

import org.junit.Before;
import org.junit.Test;

import nz.org.riskscape.dsl.Token;
import nz.org.riskscape.engine.function.FunctionResolver;
import nz.org.riskscape.engine.function.IdentifiedFunction;
import nz.org.riskscape.engine.function.OperatorResolver;
import nz.org.riskscape.engine.function.RiskscapeFunction;
import nz.org.riskscape.engine.rl.RealizationContext;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.TokenTypes;
import nz.org.riskscape.rl.ast.BinaryOperation;
import nz.org.riskscape.rl.ast.Constant;
import nz.org.riskscape.rl.ast.FunctionCall;

public class FunctionSetTest extends ProjectTest {

  FunctionSet functionSet;

  FunctionCall fc = new FunctionCall(Token.token(TokenTypes.IDENTIFIER, "foo"), Collections.emptyList());

  BinaryOperation bo = new BinaryOperation(
      Constant.string("foo"),
      Token.token(TokenTypes.AND, "and"),
      Constant.string("bar")
  );

  RealizationContext context;

  @Before
  public void setup() {
    functionSet = new DefaultFunctionSet();
    context = project.newRealizationContext();
  }

  @Test
  public void functionResolvingIsPassedThroughToFunctionResolver() throws Exception {
    IdentifiedFunction mockFunction = mock(IdentifiedFunction.class);
    when(mockFunction.getId()).thenReturn("foo");
    functionSet.add(mockFunction);
    FunctionResolver resolver = (c, f, in, a, b) -> ResultOrProblems.of(mockFunction);

    assertThat(
        functionSet.resolve(context, fc, Types.ANYTHING, Arrays.asList(Types.ANYTHING), resolver),
        Matchers.result(mockFunction));
  }

  @Test
  public void functionResolvingNotUsedIfFunctionDoesNotExist() throws Exception {
    FunctionResolver mockResolver = mock(FunctionResolver.class);

    assertFalse(functionSet.resolve(context, fc, Types.ANYTHING, Arrays.asList(Types.ANYTHING), mockResolver)
        .isPresent());
    verify(mockResolver, never()).resolve(any(), any(), any(), any(), any());
  }

  @Test
  public void firstMatchingOperatorWins() throws Exception {
    OperatorResolver operatorResolver1 = mock(OperatorResolver.class);
    OperatorResolver operatorResolver2 = mock(OperatorResolver.class);

    functionSet.insertFirst(operatorResolver2);
    functionSet.insertFirst(operatorResolver1);

    RiskscapeFunction rf1 = mock(RiskscapeFunction.class);
    RiskscapeFunction rf2 = mock(RiskscapeFunction.class);

    when(operatorResolver1.resolve(any(), any(), any(), any(), any())).thenReturn(Optional.of(rf1));
    when(operatorResolver2.resolve(any(), any(), any(), any(), any())).thenReturn(Optional.of(rf2));

    assertSame(
      rf1,
      functionSet.resolve(realizationContext, bo, Types.ANYTHING, Types.BOOLEAN, Types.BOOLEAN).orElse(null)
    );
  }

  @Test
  public void wrapsFunctionValidationErrorsWithIdentifiedFunctionContext() {
    Problem functionError = Problem.error("mock problem");

    IdentifiedFunction mockFunction = mock(IdentifiedFunction.class);
    when(mockFunction.getId()).thenReturn("foo");
    when(mockFunction.validate(realizationContext)).thenReturn(ResultOrProblems.failed(functionError));
    functionSet.add(mockFunction);

    List<Problem> problems = new ArrayList<>();
    functionSet.getAllProblems();
    functionSet.validate(bindingContext, p -> problems.add(p));

    assertThat(problems, hasSize(1));
    Problem found = problems.get(0);
    assertEquals(found.getCode().name(), "FAILED_TO_VALIDATE");
    assertThat(found.getArguments()[1], is("foo"));

    assertThat(found.getChildren(), contains(functionError));
  }

}
