/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine;

import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryFactory;
import java.io.File;
import java.util.ArrayList;
import java.util.List;
import org.geotools.api.data.FeatureSource;
import org.geotools.feature.FeatureCollection;
import org.geotools.feature.FeatureIterator;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import org.junit.Test;
import org.geotools.api.feature.Feature;
import org.geotools.api.feature.type.FeatureType;
import org.geotools.api.feature.type.GeometryDescriptor;
import org.geotools.api.referencing.crs.CoordinateReferenceSystem;

public class FeatureSourcesTest {

  GeometryFactory geomFactory = new GeometryFactory();

  @Test
  public void canOpenCsvFileContainingWktGeometries() throws Exception {
    checkForExpectedHouse(openFeatureSourceFromFile("houses-as-wkt.csv"));
  }

  @Test
  public void canOpenCsvFileContainingLatLongColumns()throws Exception {
    checkForExpectedHouse(openFeatureSourceFromFile("houses-as-lat-long.csv"));
  }

  private void checkForExpectedHouse(FeatureSource fs) throws Exception {
    List<Geometry> found = new ArrayList<>();
    FeatureCollection fc = fs.getFeatures();
    assertEquals(4, fc.size());
    FeatureType type = fs.getSchema();
    CoordinateReferenceSystem crs = type.getCoordinateReferenceSystem();
    System.out.format("%s%n", crs.toString());
    GeometryDescriptor geomDescriptor = type.getGeometryDescriptor();
    assertNotNull(type.getDescriptor("floor_level_mm"));
    assertNotNull(type.getDescriptor("replacement_cost"));
    try (FeatureIterator fi = fc.features()) {
      while (fi.hasNext()) {
        Feature f = fi.next();
        found.add((Geometry)f.getProperty(geomDescriptor.getLocalName()).getValue());
      }
    }
    assertTrue(found.contains(makePoint(15, 15)));
    assertTrue(found.contains(makePoint(25, 15)));
    assertTrue(found.contains(makePoint(17, 17)));
    assertTrue(found.contains(makePoint(25, 15)));
  }

  private Geometry makePoint(int x, int y) {
    return geomFactory.createPoint(new Coordinate(x, y));
  }

  private FeatureSource openFeatureSourceFromFile(String filename) {
    return FeatureSources.open(new File("src/test/resources/nz/org/riskscape/engine/" + filename));
  }

}
