/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.config.ini;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.net.URI;
import java.util.List;

import org.junit.Test;

import nz.org.riskscape.config.ConfigString;
import nz.org.riskscape.ini.IniFile;
import nz.org.riskscape.ini.IniParser2;

/**
 * Covers off a few integration bits between the configsection api and the raw ast
 */
public class IniConfigSectionTest {

  @Test
  public void blankValuesAreIgnored() {
    IniFile iniFile = new IniParser2().parse("""
        [section]
        foo =
        bar = baz
        """);


    IniConfigSection configSection =
        new IniConfigSection(URI.create("foo:bar"), "section", iniFile.getSections().get(0));

    assertThat(configSection.getAll("foo"), empty());
    assertThat(configSection.getAll("bar"), contains("baz"));
    assertThat(configSection.getAll("baz"), empty());
  }

  @Test
  public void keysAndValuesMaintainALinkToTheirLocation() throws Exception {
    IniFile iniFile = new IniParser2().parse("""
        [section]
        foo = bar
        """);

    URI location = URI.create("foo:bar");

    IniConfigSection configSection =
        new IniConfigSection(location, "section", iniFile.getSections().get(0));

    assertThat(
        configSection.toConfigMap(),
        allOf(
            hasEntry(ConfigString.wrap(location, "foo"), List.of(ConfigString.wrap(location, "bar")))
        )
    );
  }

  @Test
  public void leadingAndTrailingWhiteSpaceIsTrimmed() {
    IniFile iniFile = new IniParser2().parse("""
        [section]
        foo =            # empty with trailing comment
        bar = baz
        baz = bazza      # with trailing comment
        """);


    IniConfigSection configSection =
        new IniConfigSection(URI.create("foo:bar"), "section", iniFile.getSections().get(0));

    assertThat(configSection.getAll("foo"), empty());
    assertThat(configSection.getAll("bar"), contains("baz"));
    assertThat(configSection.getAll("baz"), contains("bazza"));
  }
}
