/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.config;

import static nz.org.riskscape.engine.Matchers.*;
import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.junit.Test;

import nz.org.riskscape.engine.bind.ParamProblems;
import nz.org.riskscape.engine.bind.Parameter;
import nz.org.riskscape.engine.bind.impl.NumberBinder;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problem.Severity;
import nz.org.riskscape.problem.Problems;

public class BootstrapIniSettingsTest {

  BootstrapIniSettings settings;

  List<Problem> problems = new ArrayList<>();

  Map<String, List<String>> config = new HashMap<>();

  @Test
  public void canLoadDefaultSettings() throws Exception {
    settings = build();
    assertThat(settings.getOrDefault("bool", true), is(true));
    assertThat(settings.getOrDefault("int", 789), is(789));
    assertThat(settings.getOrDefault("float", 12.3), is(12.3));
  }

  @Test
  public void canParseBasicSettings() throws Exception {
    settings = withSetting("bool", "no")
        .and("int", "123")
        .and("float", "45.6")
        .build();
    assertThat(settings.getOrDefault("bool", true), is(false));
    assertThat(settings.getOrDefault("int", 789), is(123));
    assertThat(settings.getOrDefault("float", 12.3), is(45.6));
  }

  @Test
  public void canGetWarningIfBadValueSpecified() throws Exception {
    settings = withSetting("bool", "bad")
        .and("int", "12.3")
        .build();
    assertThat(settings.getOrDefault("bool", true), is(true));
    assertThat(settings.getOrDefault("int", 789), is(789));
    assertThat(problems, contains(
        equalIgnoringChildren(Problems.foundWith(Parameter.class, "bool")
            .withSeverity(Severity.WARNING)),
        hasAncestorProblem(is(NumberBinder.PROBLEMS.numberFormatException("12.3", Integer.class)
            .withSeverity(Severity.WARNING)))
    ));
  }

  @Test
  public void canGetWarningIfTooManyValuesGiven() throws Exception {
    settings = withSetting("bool", "no")
        .and("bool", "false")
        .build();
    assertThat(settings.getOrDefault("bool", true), is(false));
    assertThat(problems, contains(
        equalIgnoringChildren(ParamProblems.get().wrongNumberGiven("bool", "1", 2)
            .withSeverity(Severity.WARNING))
    ));
  }

  private BootstrapIniSettingsTest withSetting(String key, String value) {
    List<String> currentSettings = config.getOrDefault(key, new ArrayList<>());
    currentSettings.add(value);
    config.put(key, currentSettings);
    return this;
  }

  private BootstrapIniSettingsTest and(String key, String value) {
    return withSetting(key, value);
  }

  private BootstrapIniSettings build() {
    settings = new BootstrapIniSettings(config, p -> problems.add(p));
    return settings;
  }
}
