/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.typexp;

import lombok.Getter;
import lombok.RequiredArgsConstructor;
import nz.org.riskscape.dsl.LexerException;
import nz.org.riskscape.dsl.ParseException;
import nz.org.riskscape.engine.FailedObjectException;
import nz.org.riskscape.engine.NoSuchObjectException;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.TypeInformation;
import nz.org.riskscape.engine.typeset.TypeSet;
import nz.org.riskscape.engine.typexp.AST.Symbol;

@RequiredArgsConstructor
public class DefaultTypeBuilder implements TypeBuilder {

  @Getter
  private final TypeSet typeSet;

  @Override
  public Type build(String typeExpression) {
    Parser parser = new Parser(typeExpression);
    AST type;
    try {
      type = parser.parseTypeFromExpression();
    } catch (ParseException | LexerException e) {
      throw new BadTypeExpressionException(typeExpression, e);
    }

    return expectType(type, ast -> new BadTypeExpressionException(typeExpression, "did not result in a type", null));
  }

  @Override
  public Type buildComplexType(AST.ComplexType type) {
    if (type.ident().equals("lookup")) {
      return LookupConstructor.INSTANCE.construct(this, type);
    }

    try {
      TypeInformation typeInfo = typeSet.getTypeRegistry()
          .get(type.ident());
      return typeInfo.getConstructor().construct(this, type);
    } catch (NoSuchObjectException | FailedObjectException e) {
      throw new UnknownTypeException(type);
    }
  }

  @Override
  public Type buildSimpleType(Symbol symbol) {
    try {
      TypeInformation typeInfo = typeSet.getTypeRegistry()
          .get(symbol.ident());
      return typeInfo.getConstructor().construct(this, symbol);
    } catch (NoSuchObjectException | FailedObjectException e) {
      throw new UnknownTypeException(symbol);
    }
  }

}
