/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.tiff;

import java.awt.Rectangle;
import java.awt.geom.Rectangle2D;
import java.io.File;

import org.geotools.api.coverage.CannotEvaluateException;
import org.geotools.api.coverage.PointOutsideCoverageException;
import org.geotools.api.coverage.grid.GridCoverage;
import org.geotools.api.geometry.MismatchedDimensionException;
import org.geotools.api.geometry.Position;
import org.geotools.api.referencing.operation.MathTransform2D;
import org.geotools.api.referencing.operation.TransformException;
import org.geotools.coverage.grid.GridCoordinates2D;
import org.geotools.coverage.grid.GridCoverage2D;
import org.geotools.coverage.grid.InvalidGridGeometryException;
import org.geotools.coverage.util.CoverageUtilities;
import org.geotools.geometry.Position2D;

import it.geosolutions.jaiext.range.NoDataContainer;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;

/**
 * Extends GridCoverage2D to make use of {@link SparseTIFFImageReader#isEmptyTile(int, int)}.  Using this method
 * allows us to avoid bringing empty tiles in to JAI's tile cache, which uses RAM and processing time.  Instead we can
 * short circuit the usual pixel lookup and return NO_DATA when an empty tile is queried.
 */
@Slf4j
public class SparseTiffCoverage extends GridCoverage2D {

  private final SparseTIFFImageReader imageReader;

  @Getter
  private final double[] noData;

  // profiling shows it's worth memoizing these two fields
  private final MathTransform2D crsToGrid2D;
  private final Rectangle imageBounds;

  public static GridCoverage createIfSupported(GridCoverage coverage, File file) {
    if (coverage instanceof GridCoverage2D grid2d) {
      try {
        NoDataContainer noData = CoverageUtilities.getNoDataProperty(grid2d);

        if (noData == null) {
          log.info("Sparse tiff support not enabled, missing nodata: {}", file);
          return coverage;
        }
      } catch (Exception e) {
        log.warn("Sparse tiff support not enabled, exception: {}", file, e);
        return coverage;
      }

      log.info("Using sparse tiff support for {}", file);
      return new SparseTiffCoverage(grid2d.getName(), grid2d);
    }

    log.info("Sparse tiff support not enabled for {}", file);
    return coverage;
  }

  public SparseTiffCoverage(CharSequence name, GridCoverage2D coverage) {
    super(name, coverage);

    // We need to initialize the reader with knowledge of which image from the TIFF is being sampled (TIFFs can contain
    // many actual images, and this info is not exposed in the GridCoverage2D)
    this.imageReader = SparseTIFFImageReader.initialize(this);
    this.noData = CoverageUtilities.getNoDataProperty(coverage).getAsArray();
    this.crsToGrid2D = getGridGeometry().getCRSToGrid2D();
    this.imageBounds = image.getBounds();
  }

  /**
   * Overrides the evaluate method we use from {@link GridCoverage2D} to do the isEmptyTile short circuit.
   *
   * **Important** - this method returns noData instead of throwing a {@link PointOutsideCoverageException} if the point
   * is out of bounds.  Throwing an exception is expensive and we always swallow these exceptions and turn them in to
   * no-data anyway.
   */
  @Override
  public double[] evaluate(Position coord, double[] dest) {
    Position2D translated = new Position2D();
    try {
      crsToGrid2D.transform(coord, translated);
    } catch (MismatchedDimensionException | InvalidGridGeometryException | TransformException e) {
      throw new CannotEvaluateException("Could not transform coordinate", e);
    }

    final double fx = translated.getX();
    final double fy = translated.getY();
    if (!Double.isNaN(fx) && !Double.isNaN(fy)) {
      final int x = (int) Math.round(fx);
      final int y = (int) Math.round(fy);

      if (imageBounds.contains(x, y)) {
        int tileX = image.XToTileX(x);
        int tileY = image.YToTileY(y);

        // short circuit - calling image#getTile for an uncached tile is expensive and pointless when it's full of
        // no-data
        if (imageReader.isEmptyTile(tileX, tileY)) {
          return noData;
        }

        return image.getTile(tileX, tileY).getPixel(x, y, dest);
      }
    }

    return noData;
  }

  /**
   * Overrides the evaluate method for GridCoordinates2D to do the isEmptyTile short circuit.
   * This method is used by GridTypedCoverage and SparseTIFFImageReader.
   *
   * **Important** - this method returns noData instead of throwing a {@link PointOutsideCoverageException} if the point
   * is out of bounds.  Throwing an exception is expensive and we always swallow these exceptions and turn them in to
   * no-data anyway.
   */
  @Override
  public double[] evaluate(GridCoordinates2D coord, double[] dest) {
    final int x = coord.x;
    final int y = coord.y;

    if (imageBounds.contains(x, y)) {
      int tileX = image.XToTileX(x);
      int tileY = image.YToTileY(y);

      // short circuit - calling image#getTile for an uncached tile is expensive and pointless when it's full of
      // no-data
      if (imageReader.isEmptyTile(tileX, tileY)) {
        return noData;
      }

      return image.getTile(tileX, tileY).getPixel(x, y, dest);
    } else {
      return noData;
    }
  }

  void sampleAPixelForInit() {
    Rectangle2D rectangle2d = getGridGeometry().getGridRange2D().getBounds2D();
      super.evaluate(new GridCoordinates2D((int) rectangle2d.getCenterX(), (int) rectangle2d.getCenterY()),
          new double[2]);
  }
}
