/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.task;

import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.sched.Page;

/**
 * Buffer that a {@link WorkerTask} writes output to. The {@link Tuple} output
 * is split into chunks ('pages') to allow for parallelization (i.e. different
 * threads can each work on their own page of output, which they then add to
 * this shared output buffer.
 */
public interface WritePageBuffer {

  /**
   * Adds a page of output Tuples to the buffer.
   */
  void add(Page page);

  /**
   * @return true if the WritePageBuffer has reached capacity. If so, the downstream
   * worker task needs to read pages from the buffer before more can be written.
   * Note that due to supporting multi-threaded writers, this is a polite back-off
   * that the writing worker task should honour, rather than a hard limit, i.e.
   * {@link #newPage()} and {@link #add(Page page)} will not strictly enforce the
   * capacity limit.
   */
  boolean isFull();

  /**
   * Allocates a new page to write Tuples to. Note that {@link #add(Page page)}
   * actually adds the Tuples to the buffer.
   */
  Page newPage();

  /**
   * Signals to the downstream worker task that this process has now completed
   * writing all its output to the buffer.
   */
  void markComplete();

  /**
   * @return the total number of Tuples written to this buffer (for diagnostics).
   */
  long numTuplesWritten();

  long numTuplesRead();

  int getTupleCapacity();

}
