/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.task;

import com.codahale.metrics.Meter;

import lombok.Getter;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.pipeline.RealizedStep;
import nz.org.riskscape.engine.pipeline.Sink;
import nz.org.riskscape.engine.pipeline.SinkConstructor;
import nz.org.riskscape.problem.ProblemException;

public class SinkTask extends WorkerTask {

  @Getter
  public final Sink sink;

  private final Meter full;

  public SinkTask(TaskSpec spec) throws ProblemException {
    super(spec);

    SinkConstructor constructor = (SinkConstructor) getFirstStepRealizedResult();
    this.sink = constructor.newInstance(spec.getJobContext())
        .drainWarnings(spec.getExecutionContext().getProject().getProblemSink())
        .getOrThrow();

    this.full = spec.newMetric("full", () -> new Meter());
  }

  public RealizedStep getRealizedStep() {
    return getSpec().getForSteps().get(0);
  }

  @Override
  public boolean isReadyToRun() {
    // NB the isComplete is test is here to force one last run so we can ourselves be marked as complete.  Without this,
    // the scheduler deadlocks - we really should make this a thing that the scheduler can force
    return sink.canAccept() && pageReader.hasInput() || pageReader.isComplete();
  }

  @Override
  public ReturnState run() {

    while (!pageReader.isComplete()) {
      // we peek here in case we can't successfully add it to the sink,
      // in which case we'll read the same tuple next time round
      Tuple tuple  = pageReader.peek();

      if (tuple == null) {
        return ReturnState.INPUT_EMPTY;
      }

      in.mark();
      if (!sink.accept(tuple)) {
        full.mark();
        return ReturnState.OUTPUT_FULL;
      }

      // if the tuple was accepted, we call next to consume it
      pageReader.next();
    }

    sink.finish();

    // allow any dependencies to do something with the sink.  In the past, this wasn't ever going to happen, but we now
    // use sinks to collect results in non-terminal parts of the pipeline, such as accumulation
    processingResult = sink;
    return taskComplete();
  }

  @Override
  public boolean producesResult() {
    return true;
  }

  @Override
  public String getSpecNameBrief() {
    return super.getSpecNameBrief() + "-sink";
  }
}
