/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.task;

import lombok.Getter;
import lombok.RequiredArgsConstructor;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.sched.Page;

/**
 * Writes tuples to a buffer 'page'.
 * The {@link WritePageBuffer} is potentially shared between multiple {@link WorkerTask}
 * threads, and so is thread-safe.
 * The PageWriter is not thread-safe, however, it should only ever belong to
 * a single {@link WorkerTask} (so there's no potential contention).
 * Each {@link WorkerTask} deals with its own page, which they get one at a time
 * from the {@link WritePageBuffer}. The PageWriter manages the page for a
 * {@link WorkerTask} - it tracks where in the current page the task is up to
 * and whether it needs a new one.
 * Note that because the pipeline can potentially consume a lot of memory, we want
 * to drop any references to tuple pages as soon as possible. Centralizing the page
 * reading logic helps with that.
 */
@RequiredArgsConstructor
public class PageWriter {

  private final WritePageBuffer output;
  private Page page = null;
  @Getter
  private long tuplesWritten = 0;
  @Getter
  private long pagesWritten = 0;

  /**
   * @return true if there is currently a page we're in the middle of reading.
   * Note that this is particularly important as a worker task should never
   * complete while it still has a page in progress.
   */
  public boolean hasPageInProgress() {
    return page != null;
  }

  /**
   * @return true if there's currently no space available to write new tuples to.
   */
  public boolean isFull() {
    return !hasPageInProgress() && output.isFull();
  }

  public boolean hasSpace() {
    return !isFull();
  }

  /**
   * Flushes any unwritten tuples, i.e. we have a partially written page that has
   * not been added to the WritePageBuffer yet (because the page isn't full yet). Note
   * that it is particularly important to always call `flushPage()` before the
   * worker tasks completes, so that output tuples don't get lost.
   */
  public void flushPage() {
    if (page != null) {
      output.add(page);
    }
    page = null;
  }

  /**
   * Adds a tuple to the output WritePageBuffer. Note that you should check
   * {@link PageWriter#isFull()} beforehand to avoid over-filling the output
   * buffer.
   */
  public void add(Tuple tuple) {
    if (page == null) {
      page = newPage();
    }

    page.add(tuple);
    tuplesWritten++;

    if (isCurrentPageFull()) {
      flushPage();
    }
  }

  private Page newPage() {
    pagesWritten++;
    return output.newPage();
  }

  private boolean isCurrentPageFull() {
    return page.isFull();
  }

  @Override
  public String toString() {
    String info = String.format("[Tuples: total=%d, this-task=%d, %d pages",
        output.numTuplesWritten(), tuplesWritten, pagesWritten);
    if (page != null) {
      info += ", current=" + page.toString();
    }
    if (output.isFull()) {
      info += " (buffer full)";
    }
    return info + "]";
  }
}
