/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.steps;


import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;

import nz.org.riskscape.engine.Engine;
import nz.org.riskscape.engine.bind.ParameterField;
import nz.org.riskscape.engine.pipeline.RealizationInput;
import nz.org.riskscape.engine.pipeline.Realized;
import nz.org.riskscape.engine.pipeline.RealizedStep;
import nz.org.riskscape.engine.problem.ProblemFactory;
import nz.org.riskscape.engine.problem.SeverityLevel;
import nz.org.riskscape.engine.relation.UnnestProjection;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ExpressionParser;
import nz.org.riskscape.rl.ast.Constant;
import nz.org.riskscape.rl.ast.Expression;
import nz.org.riskscape.rl.ast.ExpressionProblems;
import nz.org.riskscape.rl.ast.ListDeclaration;
import nz.org.riskscape.rl.ast.PropertyAccess;

public class UnnestStep extends BaseStep<UnnestStep.Parameters> {

  public interface LocalProblems extends ProblemFactory {
    @SeverityLevel(Problem.Severity.WARNING)
    Problem deprecatedSyntax(String given, String replacement);
  }

  public static final LocalProblems PROBLEMS = Problems.get(LocalProblems.class);

  public static class Parameters {
    @Input
    public RealizedStep input;

    @ParameterField
    public Expression toUnnest;

    @ParameterField
    public Optional<String> indexKey;

    @ParameterField
    public boolean emitEmpty = false;

    // magic field
    public RealizationInput rInput;
  }
  public UnnestStep(Engine engine) {
    super(engine);
  }

  @Override
  public ResultOrProblems<? extends Realized> realize(Parameters parameters) {

    // we use ExpressionParser.INSTANCE.toList for the convenience of handling
    // `foo` and `[foo]` the same.
    AtomicBoolean deprecatedSyntax = new AtomicBoolean(false);
    ListDeclaration toUnnestList = ExpressionParser.INSTANCE.toList(parameters.toUnnest);
    List<String> toUnnest = toUnnestList.getElements().stream()
        .map(expr -> {
          return expr.isA(Constant.class)
              .map(constant -> {
                // unnest used to accept a list of strings, eg, ['foo', 'bar']
                // when parsed as an expression they will come through as constant expressions.
                // we can pluck the text part out of those (dropping the quotes) for backward compatibility.
                deprecatedSyntax.set(true);
                return constant.getToken().value;
              })
              .orElseGet(() -> expr.toSource());
        })
        .collect(Collectors.toList());

    // Unnest will only work with PropertyAccess expressions. But Constants are also supported for now
    // for backwards compatability.
    List<Problem> toUnnestProblems = new ArrayList<>();
    for (Expression unnest: toUnnestList.getElements()) {
      if (! (unnest.isA(PropertyAccess.class).isPresent() || unnest.isA(Constant.class).isPresent())) {
        toUnnestProblems.add(ExpressionProblems.get().mismatch(unnest, PropertyAccess.class, "list"));
      }
    }
    if (!toUnnestProblems.isEmpty()) {
      return ResultOrProblems.failed(Problems.foundWith("to-unnest", toUnnestProblems));
    }

    List<Problem> warnings = new ArrayList<>();
    if (deprecatedSyntax.get()) {
      boolean isList = parameters.toUnnest.isA(ListDeclaration.class).isPresent();
      String replacementToUnnest = toUnnest.stream()
          .collect(Collectors.joining(", ", isList ? "[" : "", isList ? "]" : ""));
      warnings.add(PROBLEMS.deprecatedSyntax(String.format("unnest(to-unnest: %s)",
          parameters.toUnnest.toSource()), String.format("unnest(to-unnest: %s)", replacementToUnnest)
      ));
    }
    return new UnnestProjection(toUnnest, parameters.indexKey, parameters.emitEmpty)
        .getFlatProjector(parameters.input.getProduces())
        .withMoreProblems(warnings);
  }

}
