/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.steps;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import nz.org.riskscape.engine.Engine;
import nz.org.riskscape.engine.Project;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.bind.BindingContext;
import nz.org.riskscape.engine.bind.JavaParameterSet;
import nz.org.riskscape.engine.bind.ParameterField;
import nz.org.riskscape.engine.output.Format;
import nz.org.riskscape.engine.pipeline.RealizationInput;
import nz.org.riskscape.engine.pipeline.RealizedStep;
import nz.org.riskscape.engine.pipeline.SinkConstructor;
import nz.org.riskscape.engine.pipeline.sink.SaveSinkConstructor;
import nz.org.riskscape.engine.problem.ProblemFactory;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ast.Constant;
import nz.org.riskscape.rl.ast.Expression;
import nz.org.riskscape.rl.ast.StructDeclaration;
import nz.org.riskscape.rl.ast.StructDeclaration.Member;

/**
 * Saves results out to supported formats, relative to some storage location
 */
public class SaveStep extends BaseStep<SaveStep.Params> {

  public interface LocalProblems extends ProblemFactory {

    /**
     * When options have been included without also including the format.
     */
    Problem optionsWithNoFormat();

    Problem optionsRequired(Format format, List<String> requiredParams);
  }

  public static final LocalProblems PROBLEMS = Problems.get(LocalProblems.class);

  public static class Params {

    @ParameterField
    public Optional<String> name = Optional.empty();

    @ParameterField
    public Optional<Format> format = Optional.empty();

    @ParameterField
    public Optional<StructDeclaration> options = Optional.empty();

    @Input
    public RealizedStep step;

    public RealizationInput input;

    // set by other code to allow a ResourceOutputSink to be created without going through binding
    public Project project;

    public Project getProject() {
      if (input != null) {
        return input.getBindingContext().getProject();
      } else {
        return project;
      }
    }
  }

  public SaveStep(Engine engine) {
    super(engine);
  }

  @Override
  public ResultOrProblems<SinkConstructor> realize(Params parameters) {
    return buildFormatOptions(parameters)
        .map(options -> new SaveSinkConstructor(parameters, options));
  }

  /**
   * Build the format specific options if the format and options have both been specified.
   *
   * Options can only be added if the format is specified because the options object to bind to is
   * specific to the format. We could otherwise pass constant tuple down to the writer constructor
   * and let them bind to the specific option but that would overly complicate error reporting.
   *
   * Also note that unlike {@link FunctionCallOptions} save options do not need to be constant.
   */
  private ResultOrProblems<Optional<? extends Format.FormatOptions>> buildFormatOptions(Params parameters) {
    BindingContext context = parameters.input.getBindingContext();
    boolean noOptions = (parameters.options.isEmpty() || parameters.options.get().getMembers().isEmpty());
    if (parameters.format.isEmpty()) {
      if (noOptions) {
        return ResultOrProblems.of(Optional.empty());
      } else {
        // you can't have format options if you don't tell us what the format is
        return ResultOrProblems.failed(PROBLEMS.optionsWithNoFormat());
      }
    }

    Format format = parameters.format.get();
    @SuppressWarnings("unchecked")
    JavaParameterSet<? extends Format.FormatOptions> parameterSet =
            JavaParameterSet.fromBindingClass(format.getWriterOptionsClass());
    List<String> required = format.getRequiredOptions(context);

    if (noOptions) {
      // check if format requires options but none were provided
      if (!required.isEmpty()) {
        return ResultOrProblems.failed(PROBLEMS.optionsRequired(format, required));
      }

      // no options provided, nothing more to do here.
      return ResultOrProblems.of(Optional.empty());
    }

    Map<String, List<?>> paramMap = new HashMap<>();
    for (Member member : parameters.options.get().getMembers()) {
      String memberName = member.getName().orElseThrow();
      Object value = member.getExpression();
      if (value instanceof Constant constant) {
        if (parameterSet.contains(memberName)
            && !Expression.class.isAssignableFrom(parameterSet.get(memberName).getType())) {
        // if the value is a constant expression and isn't going to an expression type then we
        // extract the constant value to bind that.
        value = parameters.input.getRealizationContext().getExpressionRealizer().realizeConstant(constant)
            .get()
            .evaluate(Tuple.EMPTY_TUPLE);
        }
      }
      if (value instanceof List list) {
        paramMap.put(memberName, list);
      } else {
        paramMap.put(memberName, List.of(value));
      }
    }

    return format.buildOptions(paramMap, context, parameters.step.getProduces())
        .map(built -> Optional.of(built));
  }

}
