/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.steps;

import java.util.List;
import java.util.Optional;

import com.google.common.collect.Lists;

import nz.org.riskscape.engine.Engine;
import nz.org.riskscape.engine.bind.ParameterField;
import nz.org.riskscape.engine.join.DefaultJoiner;
import nz.org.riskscape.engine.join.ExpressionJoinCondition;
import nz.org.riskscape.engine.join.HashIndexer;
import nz.org.riskscape.engine.join.JoinCondition;
import nz.org.riskscape.engine.join.JoinIndexer.Constructor;
import nz.org.riskscape.engine.join.JoinType;
import nz.org.riskscape.engine.join.Joiner;
import nz.org.riskscape.engine.join.NoIndexIndexer;
import nz.org.riskscape.engine.join.SpatialIndexer;
import nz.org.riskscape.engine.pipeline.RealizationInput;
import nz.org.riskscape.engine.pipeline.Realized;
import nz.org.riskscape.engine.pipeline.RealizedStep;
import nz.org.riskscape.engine.rl.ExpressionRealizer;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ast.Expression;

public class JoinStep extends BaseStep<JoinStep.Parameters> {

  /**
   * {@link Constructor}s in priority order.
   */
  public static final List<Constructor> INDEX_CONSTRUCTORS = Lists.newArrayList(
      SpatialIndexer.CONSTRUCTOR,
      HashIndexer.CONSTRUCTOR,
      NoIndexIndexer.CONSTRUCTOR
  );

  public static class Parameters {

    @ParameterField
    public Expression on;

    @ParameterField
    public JoinType joinType = JoinType.INNER;

    @ParameterField
    public Optional<Integer> initialIndexSize;

    @Input
    public RealizedStep lhs;

    @Input
    public RealizedStep rhs;

    //Injected field
    public RealizationInput rInput;

  }

  public JoinStep(Engine engine) {
    super(engine);
  }

  @Override
  public ResultOrProblems<? extends Realized> realize(Parameters parameters) {
    ExpressionRealizer expressionRealizer = parameters.rInput.getExecutionContext().getExpressionRealizer();

    //Seed the join with a cartesian join condition. This is to get the shape of the joined data
    return Joiner.build(
        parameters.lhs.getProduces(),
        parameters.rhs.getProduces(),
        JoinCondition.CARTESIAN,
        parameters.joinType).flatMap(join -> {

      //now realize the expression join condition for the joined data
      return ExpressionJoinCondition.realize(parameters.on, join.getJoinedType(), expressionRealizer)
          .flatMap(on -> {
            return ResultOrProblems.of(
                //and update join with the realized expression join condition
                new DefaultJoiner(join.withJoinConditon(on), INDEX_CONSTRUCTORS, expressionRealizer,
                    parameters.initialIndexSize));
          });
      });
  }

}
