/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.steps;

import java.util.Optional;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.atomic.AtomicBoolean;

import nz.org.riskscape.engine.Engine;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.bind.ParameterField;
import nz.org.riskscape.engine.output.PipelineJobContext;
import nz.org.riskscape.engine.pipeline.Realized;
import nz.org.riskscape.engine.pipeline.Sink;
import nz.org.riskscape.engine.pipeline.SinkConstructor;
import nz.org.riskscape.engine.pipeline.TupleInput;
import nz.org.riskscape.engine.projection.AsyncProjector;
import nz.org.riskscape.engine.projection.Projector;
import nz.org.riskscape.engine.projection.SelectProjector;
import nz.org.riskscape.engine.steps.ProjectionStep.ProjectionParameters;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ast.Expression;

/**
 * A version of a standard select that exposes itself as an {@link AsyncProjector} for testing purposes.
 */
public class AsyncSelectStep extends BaseStep<SelectStep.Parameters> {

  public static class Parameters extends ProjectionParameters {

    @ParameterField()
    Expression select;
  }

  public AsyncSelectStep(Engine engine) {
    super(engine);
  }

  @Override
  public ResultOrProblems<? extends Realized> realize(SelectStep.Parameters parameters) {
      return SelectProjector.realize(parameters.input.getProduces(), parameters.select,
          parameters.rInput.getExecutionContext().getExpressionRealizer()).map(this::
            buildAsyncProjector);
  }

  private AsyncProjector buildAsyncProjector(Projector projector) {

    LinkedBlockingQueue<Tuple> queue = new LinkedBlockingQueue<>(10);
    AtomicBoolean finished = new AtomicBoolean(false);

    return new AsyncProjector() {

      @Override
      public Struct getProducedType() {
        return projector.getProducedType();
      }

      @Override
      public SinkConstructor getOutput() {
        return new SinkConstructor() {

          @Override
          public ResultOrProblems<Sink> newInstance(PipelineJobContext context) {
            return ResultOrProblems.of(new Sink() {
              @Override
              public void finish() {
                finished.set(true);
              }

              @Override
              public boolean accept(Tuple tuple) {
                return queue.offer(tuple);
              }
            });
          }
        };
      }

      @Override
      public TupleInput getInput() {
        return new TupleInput() {

          @Override
          public Struct getProducedType() {
            return projector.getProducedType();
          }

          @Override
          public Tuple poll() {
            return queue.poll();
          }

          @Override
          public boolean isComplete() {
            return finished.get();
          }

          @Override
          public boolean isAsync() {
            return false;
          }

          @Override
          public Optional<Long> size() {
            return Optional.empty();
          }
        };
      }
    };
  }
}
