/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.sched;

import java.util.Arrays;
import java.util.List;
import java.util.Optional;

import lombok.AccessLevel;
import lombok.Getter;
import nz.org.riskscape.config.BootstrapIniSettings;

class SchedulerParams {

  public static final String PAGE_SIZE_SETTING = "engine.page-size";
  public static final String TUPLES_PER_TASK_SETTING = "engine.tuples-per-task";

  public static final List<String> SETTINGS_KEYS = Arrays.asList(
      PAGE_SIZE_SETTING,
      TUPLES_PER_TASK_SETTING
  );

  // We need some back-pressure to ensure that the tuples queued up for
  // processing by the various pipeline steps doesn't exhaust memory.
  // Note that this only limits the tuples sitting in a buffer
  // somewhere - it doesn't include the tuple pages that are actively
  // being processed by a task.
  public static final int DEFAULT_MAX_TUPLES_QUEUED = 16384;

  // if we allow too many tuples to be queued up on the buffers, then
  // it can cause OOM errors. To avoid this, we place an upper limit on
  // the total size of all tuple buffers in the pipeline. Note this only
  // covers *queued* tuples, not ones actively being processed by a task
  @Getter(AccessLevel.PACKAGE)
  private int maxTuplesQueued = DEFAULT_MAX_TUPLES_QUEUED;

  @Getter(AccessLevel.PACKAGE)
  private Optional<Integer> pageSize;

  @Getter(AccessLevel.PACKAGE)
  private Optional<Integer> tuplesPerTask;

  @Getter(AccessLevel.PACKAGE)
  private final int numThreads;

  SchedulerParams(int numThreads) {
    this(numThreads, BootstrapIniSettings.NONE);
  }

  SchedulerParams(int numThreads, BootstrapIniSettings settings) {
    this.numThreads = numThreads;
    int config = settings.getOrDefault(PAGE_SIZE_SETTING, 0);
    this.pageSize = Optional.ofNullable(config > 0 ? config : null);

    config = settings.getOrDefault(TUPLES_PER_TASK_SETTING, 0);
    this.tuplesPerTask = Optional.ofNullable(config > 0 ? config : null);
  }

  int getMaxThreadsPerTask() {
    // give it as many threads as we can and hope for the best
    return numThreads;
  }


}
