/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.sched;

import java.util.List;
import java.util.Map;
import java.util.concurrent.Future;
import java.util.stream.Collectors;

import nz.org.riskscape.config.BootstrapIniSettings;
import nz.org.riskscape.engine.Engine;
import nz.org.riskscape.engine.output.PipelineJobContext;
import nz.org.riskscape.engine.pipeline.DefaultPipelineExecutor;
import nz.org.riskscape.engine.pipeline.ExecutionResult;
import nz.org.riskscape.engine.util.Pair;

/**
 * Executes a pipeline by breaking the steps down into tasks, and then using a
 * Scheduler to manage running the tasks in parallel where possible. First we
 * take the pipeline and group together steps that can be run together
 * (LinkedSteps). When then take the LinkedSteps and convert them to tasks
 * (TaskSpec) that the Scheduler can run.
 */
public class SchedulerBasedExecutor extends DefaultPipelineExecutor {

  private Scheduler scheduler;

  public SchedulerBasedExecutor(Engine engine) {
    super(engine);
  }

  private synchronized Scheduler getScheduler() {
    // We create the scheduler just before we first try to use it.
    // By this point we should have settled on how many threads to give it.
    if (scheduler == null) {
      this.scheduler = new Scheduler(getSchedulerParams(), engine.getProblemSink());
    }
    return scheduler;
  }

  private SchedulerParams getSchedulerParams() {
    // extract any scheduler specific settings.ini config from the engine
    Map<String, List<String>> schedulerSettings =
        SchedulerParams.SETTINGS_KEYS.stream()
        .map(key -> Pair.of(key, engine.getSetting(key)))
        .filter(pair -> !pair.getRight().isEmpty())
        .collect(Collectors.toMap(Pair::getLeft, Pair::getRight));

    return new SchedulerParams(getNumThreads(),
        new BootstrapIniSettings(schedulerSettings, engine.getProblemSink()));
  }

  @Override
  public Future<ExecutionResult> execute(PipelineJobContext jobContext) {
    return getScheduler().queueJob(jobContext);
  }
}
