/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.rl.agg;

import java.util.List;

import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.rl.RealizationContext;
import nz.org.riskscape.engine.rl.agg.ListCollecting.ListProcessor;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.RSList;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.problem.ProblemException;
import nz.org.riskscape.rl.ast.FunctionCall;

public class StandardDeviationAggregationFunction extends ListCollecting {

  public static final Struct RESULT_TYPE = Struct.of("mean", Types.FLOATING, "stddev", Types.FLOATING);

  public static final int SQUARED = 2;
  // otherwise stddev is meaningless
  public static final int MIN_VALUES_REQUIRED = 2;

  @Override
  protected ListProcessor buildProcessor(RSList listType, RealizationContext context, Type inputType,
      FunctionCall fc) throws ProblemException {

    if (!listType.getMemberType().isNumeric()) {
      throw new ProblemException(LocalProblems.get().valueNotNumeric("stddev", listType.getMemberType()));
    }
    return new ListProcessor(Nullable.of(RESULT_TYPE), StandardDeviationAggregationFunction::pickStddev);
  }

  private static Object pickStddev(List values) {
    @SuppressWarnings("unchecked")
    List<Number> numbers = (List<Number>) values;
    if (numbers.size() < MIN_VALUES_REQUIRED) {
      return null;
    }

    double total = 0;
    for (Number value : numbers) {
      total += value.doubleValue();
    }
    double mean = total / numbers.size();

    double sum = 0;
    for (Number value : numbers) {
      sum += Math.pow(value.doubleValue() - mean, SQUARED);
    }

    double stddev = Math.sqrt(sum / (numbers.size() - 1));

    return Tuple.ofValues(RESULT_TYPE, mean, stddev);
  }

}
