/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.rl.agg;

import java.util.List;
import java.util.stream.Collectors;

import com.google.common.collect.Streams;

import lombok.RequiredArgsConstructor;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.rl.RealizedExpression;
import nz.org.riskscape.engine.types.Struct;

@RequiredArgsConstructor
public class ExpressionAccumulator implements Accumulator {

  private final List<Accumulator> children;
  private final RealizedExpression processExpression;

  @Override
  public void accumulate(Object input) {
    for (Accumulator child : children) {
      child.accumulate(input);
    }
  }

  @Override
  public Accumulator combine(Accumulator rhs) {
    ExpressionAccumulator other = (ExpressionAccumulator) rhs;
    List<Accumulator> combinedChildren =
        Streams.zip(children.stream(), other.children.stream(), (l, r) -> l.combine(r)).collect(Collectors.toList());

    return new ExpressionAccumulator(combinedChildren, processExpression);
  }

  @Override
  public Object process() {
    Tuple input = new Tuple(processExpression.getInputType().find(Struct.class).get());
    for (int index = 0; index < children.size(); index++) {
      input.set(index, children.get(index).process());
    }

    return processExpression.evaluate(input);
  }

  @Override
  public boolean isEmpty() {
    // we are empty if all our children are empty
    for (Accumulator accumulator : children) {
      if (!accumulator.isEmpty()) {
        return false;
      }
    }

    return true;
  }

}
