/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.rl;

import lombok.Data;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.types.LambdaType;
import nz.org.riskscape.engine.types.ScopedLambdaType;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Struct.StructMember;
import nz.org.riskscape.rl.ast.Lambda;

/**
 * The result of calling {@link RealizedExpression#evaluate(Object)} on a {@link Lambda} that has been
 * realized.  This value can be passed to the user of the expression to execute it (or whatever)
 */
@Data
public class ScopedLambdaExpression {

  /**
   * The expression that has been returned.  At the moment, RiskScape does not support 'dynamic expressions', so this
   * will always be the same expression across evaluations of an expression
   */
  private final Lambda expression;

  /**
   * The scope that this expression closes over.  This should be the same object as the type of
   * {@link ScopedLambdaType#getScopeType()} for the {@link LambdaType} that the function is
   * realized against.  That means that any struct member from the scope type can be used to access this tuple.
   */
  private final Tuple scope;

  /**
   * Build a tuple to use as the calling scope, i.e. what you pass to
   * {@link RealizedExpression#evaluate(Object)} for this lambda expression.
   * The calling scope combines this expression's closed over scope combined with
   * the actual argument values passed to the lambda.
   *
   * @param newScopeType The type of the resulting tuple, typically built via
   * {@link ScopedLambdaType#buildCallingScopeType(nz.org.riskscape.engine.types.Type...)}
   * @param lambdaValues The arguments supplied to the expression to call it with,
   * e.g. the values of x and y from `(x, y) -> x + y`
   * @return the combination of {@link #getScope()} and the lambda arguments
   */
  public Tuple buildCallingScope(Struct newScopeType, Object... lambdaValues) {
    // copy the original input scope we have
    Tuple newScope = new Tuple(newScopeType);
    newScope.setAll(scope);

    // replace the lambda arg with the actual values to use
    for (int i = 0; i < lambdaValues.length; i++) {
      String lambdaArgName = expression.getArguments().get(i).getValue();
      StructMember itemInScope = newScopeType.getEntry(lambdaArgName);
      newScope.set(itemInScope, lambdaValues[i]);
    }
    return newScope;
  }
}
