/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.rl;

import java.util.List;

import nz.org.riskscape.engine.ArgsProblems;
import nz.org.riskscape.engine.function.ArgumentList;
import nz.org.riskscape.engine.function.FunctionMetadata;
import nz.org.riskscape.engine.function.IdentifiedFunction;
import nz.org.riskscape.engine.function.IdentifiedFunction.Category;
import nz.org.riskscape.engine.function.RiskscapeFunction;
import nz.org.riskscape.engine.function.UntypedFunction;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.ScopedLambdaType;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.engine.types.eqrule.Coercer;
import nz.org.riskscape.problem.ProblemException;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ast.ExpressionProblems;
import nz.org.riskscape.rl.ast.FunctionCall;
import nz.org.riskscape.rl.ast.FunctionCall.Argument;
import nz.org.riskscape.rl.ast.Lambda;

/**
 * Implementation of https://docs.geoserver.org/latest/en/user/filter/function_reference.html#control-functions
  */
public class IfThenElse implements RealizableFunction {

  public static final ArgumentList ARGUMENTS = ArgumentList.create(
      "condition", Nullable.BOOLEAN,
      "then", Nullable.ANYTHING,
      "else", Nullable.ANYTHING
  );

  public static final IdentifiedFunction IF = asFunction("if", "");
  public static final IdentifiedFunction IF_THEN_ELSE = asFunction("if_then_else", "");

  private static IdentifiedFunction asFunction(String id, String description) {
    FunctionMetadata metadata = new FunctionMetadata(id, ARGUMENTS, Types.ANYTHING, description,
        Category.LOGICAL, RiskscapeFunction.BUILT_IN);
    return RealizableFunction.identified(new IfThenElse(), metadata);
  }

  @Override
  public ResultOrProblems<RiskscapeFunction> realize(
      RealizationContext context,
      FunctionCall functionCall,
      List<Type> givenTypes
  ) {
    if (givenTypes.size() < 2 || givenTypes.size() > ARGUMENTS.size()) {
      return ResultOrProblems.failed(
          Problems.get(ArgsProblems.class).wrongNumberRange(2, ARGUMENTS.size(), givenTypes.size()));
    }

    return ProblemException.catching(() -> {
      ARGUMENTS.getRequiredArgument(functionCall, "condition").getOrThrow();
      ARGUMENTS.getRequiredArgument(functionCall, "then").getOrThrow();

      // we use a coercer for the test argument. this handles cases where the test is formed from a
      // comparison operator against two single attribute tuples.
      Coercer testCoercer = context.getTypeSet()
          .findEquivalenceCoercer(Nullable.strip(givenTypes.get(0)), Types.BOOLEAN)
          .orElseThrow(() -> new ProblemException(ArgsProblems.mismatch(ARGUMENTS.get(0), givenTypes.get(0))));

      RealizedExpression thenLambda = realizeLambdaExpressions(givenTypes.get(1),
          functionCall.getArguments().get(1), context);
      Type thenType = thenLambda != null ? thenLambda.getResultType() : givenTypes.get(1);

      RealizedExpression elseLambda = null;
      Type returnType;
      if (givenTypes.size() == 2) {
        // no else condition given. we'll be returning then or null
        returnType = Nullable.of(thenType);
      } else {
        elseLambda = realizeLambdaExpressions(givenTypes.get(2),
            functionCall.getArguments().get(2), context);
        Type elseType = elseLambda != null ? elseLambda.getResultType() : givenTypes.get(2);

        returnType = context.getTypeSet().computeAncestorNoConversion(thenType, elseType);
      }

      return RiskscapeFunction.create(
          this, givenTypes, returnType,
          buildFunction(testCoercer, thenLambda, elseLambda),
          // make sure these are cleaned up if realized
          thenLambda, elseLambda
      );
    });
  }

  /**
   * Realizes the lambda expression if one was provided.
   *
   * The then/else arguments do not have to be lambda expressions. But if they are they will be
   * validated (don't accept any args) and realized for later use.
   *
   * @return realized lambda expression or null if provided arg is not a lambda
   */
  private RealizedExpression realizeLambdaExpressions(Type givenType, Argument arg, RealizationContext context)
      throws ProblemException {
    ScopedLambdaType lambdaType = givenType.find(ScopedLambdaType.class).orElse(null);
    if (lambdaType == null) {
      // the argument is not a lambda expression so bail from here (lambdas are optional)
      return null;
    }
    if (lambdaType.getArity() != 0) {
      throw new ProblemException(ExpressionProblems.get().lambdaArityError(
          arg.getExpression(),
          lambdaType.getArity(),
          0
      ));
    }

    Lambda lambda = (Lambda) arg.getExpression();
    return context.getExpressionRealizer()
        .realize(lambdaType.getScopeType(), lambda.getExpression())
        .getOrThrow();
  }

  /**
   * Return an UntypedFunction for the if else, with optional then and else expressions
   */
  private UntypedFunction buildFunction(
      Coercer testCoercer,
      RealizedExpression thenLambda,
      RealizedExpression elseLambda
  ) {
    return (args) -> {
      Object condition = args.get(0);

      if (condition != null && testCoercer.apply(condition).equals(Boolean.TRUE)) {
        if (thenLambda == null) {
          return args.get(1);
        }
        ScopedLambdaExpression givenLambda = (ScopedLambdaExpression) args.get(1);
        return thenLambda.evaluate(givenLambda.getScope());
      } else {
        if (args.size() == 2) {
          // there is no else argument so we just return null
          return null;
        }
        if (elseLambda == null) {
          return args.get(2);
        }
        ScopedLambdaExpression givenLambda = (ScopedLambdaExpression) args.get(2);
        return elseLambda.evaluate(givenLambda.getScope());
      }
    };
  }
}
