/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */

package nz.org.riskscape.engine.rl;

import java.util.List;
import java.util.Optional;

import com.google.common.collect.Lists;

import lombok.EqualsAndHashCode;
import lombok.Getter;

import nz.org.riskscape.dsl.Token;
import nz.org.riskscape.rl.ExpressionParser;
import nz.org.riskscape.rl.TokenTypes;
import nz.org.riskscape.rl.ast.Expression;
import nz.org.riskscape.rl.ast.FunctionCall;
import nz.org.riskscape.rl.ast.FunctionCall.Argument;

/**
 * Wraps a {@link FunctionCall}  and provides conveniences for updating the function call arguments.
 *
 * Exists primarily for binding purposes.
 */
@EqualsAndHashCode
public class FunctionCallPrototype {

  @Getter
  private final FunctionCall prototype;
  @Getter
  private final List<Argument> newArguments;

  public FunctionCallPrototype(FunctionCall prototype) {
    this.prototype = prototype;
    this.newArguments = Lists.newArrayList(prototype.getArguments());
  }

  /**
   * Sets or replaces argument i with the {@link Expression} produced from argFormat and formatArgs.
   * If the expression is invalid then exceptions will be thrown from {@link ExpressionParser#parse(java.lang.String)}.
   *
   * Arguments must set in a manner that will not create a hole in the argument list at any time else
   * {@link IllegalArgumentException} will result. That is it is not possible to set arg 2 if arg 1
   * does not yet exist.
   *
   * @param i                        index of the argument to set (zero based)
   * @param argFormat                string format of expression to set argument with
   * @param formatArgs               args (if any) required by argFormat
   * @return FunctionCallPrototype   Expression with the replaced/set argument
   */
  public FunctionCallPrototype setArgument(int i, String argFormat, Object... formatArgs) {
    return setArgument(i, Optional.empty(), argFormat, formatArgs);
  }

  /**
   * Sets or replaces argument i with the {@link Expression} produced from argFormat and formatArgs.
   * If the expression is invalid then exceptions will be thrown from {@link ExpressionParser#parse(java.lang.String)}.
   *
   * Arguments must set in a manner that will not create a hole in the argument list at any time else
   * {@link IllegalArgumentException} will result. That is it is not possible to set arg 2 if arg 1
   * does not yet exist.
   *
   * @param i                        index of the argument to set (zero based)
   * @param argumentName             optional argument name
   * @param argFormat                string format of expression to set argument with
   * @param formatArgs               args (if any) required by argFormat
   * @return FunctionCallPrototype   Expression with the replaced/set argument (with optional arg name)
   */
  public FunctionCallPrototype setArgument(int i, Optional<String> argumentName,
      String argFormat, Object... formatArgs) {

    Expression valueExpression = ExpressionParser.INSTANCE.parse(String.format(argFormat, formatArgs));
    Argument newValueArg = new Argument(valueExpression,
        argumentName.map(name -> Token.token(TokenTypes.IDENTIFIER, name)));

    if (newArguments.size() > i) {
      // the arg list is big enough already, replace what is there.
      newArguments.set(i, newValueArg);
    } else if (newArguments.size() == i) {
      newArguments.add(newValueArg);
    } else {
      throw new IllegalArgumentException(String.format("Missing argument %d", i -1));
    }
    return this;
  }

  /**
   * Same as {@link #setArgument(int, java.lang.String, java.lang.Object...) } for argument 0.
   */
  public FunctionCallPrototype setFirstArgument(String argFormat, Object... formatArgs) {
    return setArgument(0, argFormat, formatArgs);
  }

  public FunctionCallPrototype addArgument(Optional<String> argumentName,
      String argFormat, Object... formatArgs) {
    return setArgument(newArguments.size(), argumentName, argFormat, formatArgs);
  }

  public FunctionCall build() {
    return new FunctionCall(prototype.getIdentifier(), newArguments);
  }

}
