/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.rl;

import org.locationtech.jts.geom.TopologyException;

import lombok.Getter;
import nz.org.riskscape.engine.GeometryProblems;
import nz.org.riskscape.engine.RiskscapeException;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problems;

/**
 * Thrown by {@link RealizedExpression} when {@link RealizedExpression#evaluate(Object)} fails
 *
 */
public class EvalException extends RiskscapeException {

  @Getter
  private final RealizedExpression realizedExpression;
  @Getter
  private final Object input;

  private static Problem makeProblem(String message, Throwable cause) {
    Problem problem = Problem.error(message).withException(cause);
    Throwable rootCause = getRootCause(cause);

    // TopologyExceptions generally mean the input Geometry was invalid.
    // This is a bit of a kludge to get a more meaning message back to the user.
    // Refer GL#283 for a nicer way of detecting this type of problem.
    if (rootCause instanceof TopologyException) {
      problem = problem.withChildren(GeometryProblems.get().topologyExceptionTip());
    } else {
      // try to nest any problems that might be present in the exception, as it's a
      // little more helpful than trying to decipher a large stack trace
      problem = problem.withChildren(Problems.caught(rootCause));
    }
    return problem;
  }

  public EvalException(String message, Throwable cause, RealizedExpression re, Object input) {
    super(makeProblem(message, cause), cause);
    this.realizedExpression = re;
    this.input = input;
  }

  public EvalException(Throwable cause, RealizedExpression re, Object input) {
    this("Failed to evaluate `" + re.getExpression().toSource() + "`", cause, re, input);
  }

  /**
   * @return the first exception that isn't an EvalException.  This is useful because the chain of EvalExceptions
   * is usually quite deep - and so this will be the first exception that gives the user an idea of what failed, rather
   * than where it failed.
   */
  public Throwable getRootCause() {
    return getRootCause(this);
  }

  private static Throwable getRootCause(Throwable startAt) {
    Throwable ptr = startAt;
    while (ptr.getCause() != null && ptr instanceof EvalException) {
      ptr = ptr.getCause();
    }
    return ptr;
  }


}
