/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.rl;


import nz.org.riskscape.engine.RiskscapeException;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.TokenTypes;
import nz.org.riskscape.rl.ast.BinaryOperation;
import nz.org.riskscape.rl.ast.Expression;
import nz.org.riskscape.rl.ast.ExpressionConverter;
import nz.org.riskscape.rl.ast.PropertyAccess;

import lombok.RequiredArgsConstructor;

/**
 * Utility allowing boolean yielding expressions to be realized against a {@link Struct} type that may
 * not contain all referenced properties.
 *
 * Important to not that returned expression may have parts of the expression removed. It is important
 * that the full expression be applied to achieve the desired filtering.
 */
public class BooleanExpressionSplitter {

  @RequiredArgsConstructor
  private static class Data {

    final Struct inputType;
    final ExpressionRealizer expressionRealizer;
  }

  private static class Splitter<T extends Data> extends ExpressionConverter<T> {

    @Override
    public Expression visit(PropertyAccess expression, T data) {
      ResultOrProblems<RealizedExpression> re = data.expressionRealizer.realize(data.inputType, expression);
      if (re.hasErrors()) {
        //If the property access cannot be realized to throw an exception so the property access expression
        //is not included in any resulting expression
        throw new RiskscapeException(String.format("Could not apply '%s' to type %s",
            expression.toSource(), data.inputType));
      }
      return expression;
    }

    @Override
    public Expression visit(BinaryOperation expression, T data) {
      Expression lhs = null;
      Expression rhs = null;
      try {
        lhs = expression.getLhs().accept(this, data);
      } catch (RiskscapeException e) {
        if (expression.getOperator().type != TokenTypes.KEYWORD_AND) {
          //We cannot use any part of this with or, so rethrow the exception
          throw e;
        }
      }
      try {
        rhs = expression.getRhs().accept(this, data);
      } catch (RiskscapeException e) {
        if (expression.getOperator().type != TokenTypes.KEYWORD_AND || lhs == null) {
          //We cannot use any part of this with or, so rethrow the exception
          throw e;
        }
      }
      if (lhs != null && rhs != null) {
        return new BinaryOperation(lhs, expression.getOperator(), rhs);
      } else if (lhs != null) {
        return lhs;
      }
      return rhs;
    }

  }

  /**
   * Reduces boolean expressions so they can be applied to inputType allowing as much filtering to be
   * applied as possible.
   *
   * Any expression segments that access properties not contained in inputType will be removed. This
   * may also remove expression segments to do exist where these could affect the results, eg
   * 'known.attr OR unknown.attr' would result in no expression being realized.
   *
   * The resulting expression may be used to reduce data but it is important that the full boolean
   * expression is applied to the whole data set to ensure correct results.
   *
   * @param expression to split
   * @param inputType type expression will be applied to
   * @param expressionRealizer realizer of expressions
   * @return realized expression for properties found in input type
   */
  public static ResultOrProblems<RealizedExpression> splitAndRealize(Expression expression, Struct inputType,
      ExpressionRealizer expressionRealizer) {
    try {
      @SuppressWarnings("unchecked")
      Expression converted = (Expression) expression.accept(new Splitter(), new Data(inputType, expressionRealizer));
      if (converted != null) {
        return expressionRealizer.realize(inputType, converted).flatMap(re -> {
          if (Nullable.strip(re.getResultType()) != Types.BOOLEAN) {
            return ResultOrProblems.failed(Problem.error("Expected %s from '%s' but found %s",
                Types.BOOLEAN, expression.toSource(), re.getResultType()));
          }
          return ResultOrProblems.of(re);
        });
      }
    } catch (RiskscapeException e) {
      //Fall though to failure below.
    }
    return ResultOrProblems.failed(Problem.error("Could not apply '%s' to type %s", expression.toSource(), inputType));
  }

}
