/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.projection;

import java.util.List;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;

import lombok.Getter;
import lombok.RequiredArgsConstructor;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Struct.StructBuilder;
import nz.org.riskscape.engine.types.Struct.StructMember;
import nz.org.riskscape.problem.ResultOrProblems;

/**
 * Projects tuples where nested tuples are flattened in to a new attribute, according to a specific join character,
 * e.g:
 *
 *  ```
 *  {a: {b: Text, c: Text}, {d: {e: Integer}}}
 *  ->
 *  {a.b: Text, a.c: Text, d.e: Integer}
 *  ```
 **/
public class FlattenProjection implements Projection {

  @RequiredArgsConstructor
  private class StructFlattener implements Projector {
    @Getter
    public final Struct sourceType;
    public final Struct flattenedType;
    public final List<List<StructMember>> expressions;

    @Override
    public Struct getProducedType() {
      return flattenedType;
    }

    @Override
    public Tuple apply(Tuple t) {
      Tuple mapped = new Tuple(flattenedType);
      for (int i = 0; i < expressions.size(); i++) {
        List<StructMember> indices = expressions.get(i);
        StructMember targetEntry = flattenedType.getMembers().get(i);
        Object selected;
        Tuple target = t;
        if (indices.size() > 0) {
          for (StructMember entry : indices.subList(0, indices.size() - 1)) {
            if (target != null) {
              target = target.fetch(entry);
            } else {
              break;
            }
          }
        }

        // TODO warn on non-nullable type being in tuple?
        if (target != null) {
          selected = target.fetch(indices.get(indices.size() - 1));
          mapped.set(targetEntry, selected);
        } else {
          mapped.remove(targetEntry);
        }
      }
      return mapped;
    }
  }

  private final String joinCharacter;

  public FlattenProjection(String joinWith) {
    this.joinCharacter = joinWith;
  }

  @Override
  public ResultOrProblems<Struct> projectType(Struct source) {
    return ResultOrProblems.of(structFlattener(source).flattenedType);
  }

  @Override
  public ResultOrProblems<Projector> getProjectionFunction(Struct sourceType) {
    return ResultOrProblems.of(this.structFlattener(sourceType));
  }


  protected StructFlattener structFlattener(Struct source) {
    StructBuilder builder = new StructBuilder();
    List<StructMember> stack = ImmutableList.of();
    List<List<StructMember>> expressions = Lists.newArrayList();
    flattenVisitStruct(expressions, builder, source, stack, joinCharacter, false);

    return new StructFlattener(source, builder.build(), expressions);
  }


  protected void flattenVisitStruct(List<List<StructMember>> expressions, StructBuilder copyTo,
      Struct struct, List<StructMember> stack, String joinWith, boolean nullable) {

    for (StructMember member : struct.getMembers()) {
      ImmutableList.Builder<StructMember> stackBuilder = ImmutableList.builder();
      stackBuilder.addAll(stack);
      stackBuilder.add(member);
      List<StructMember> newStack = stackBuilder.build();

      if (Nullable.unwrap(member.getType()) instanceof Struct) {
        Struct child = (Struct) Nullable.unwrap(member.getType());
        flattenVisitStruct(expressions, copyTo, child, newStack, joinWith, nullable || Nullable.any(member.getType()));
      } else {
        expressions.add(newStack);
        // XXX there probably more efficient ways of doing this...
        String prefix = newStack.stream().map(StructMember::getKey).collect(Collectors.joining(joinWith));
        copyTo.add(prefix, Nullable.ifTrue(nullable, member.getType()));
      }
    }
  }

}
