/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */

package nz.org.riskscape.engine.projection;

import lombok.AccessLevel;
import lombok.RequiredArgsConstructor;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.rl.RealizationContext;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.TypeProblems;
import nz.org.riskscape.engine.types.eqrule.Coercer;
import nz.org.riskscape.problem.ProblemException;
import nz.org.riskscape.problem.ResultOrProblems;

/**
 * A {@link Projection} that will defer to a {@link Coercer} to map tuples from the source to target type.
 */
@RequiredArgsConstructor(access = AccessLevel.PRIVATE)
public class CoercerProjection implements Projection {

  /**
   * @return a coercer projection to map source to target or problems is mapping is not possible
   */
  public static ResultOrProblems<CoercerProjection> mapping(Struct source, Struct target, RealizationContext context) {
    return ProblemException.catching(() -> {
      Coercer sourceToTarget = context.getTypeSet().findEquivalenceCoercer(source, target)
          .orElseThrow(() -> new ProblemException(TypeProblems.get().couldNotCoerce(source, target)));
      return new CoercerProjection(source, target, sourceToTarget);
    });
  }

  protected class CoercerProjector implements Projector {

    @Override
    public Struct getProducedType() {
      return producedType;
    }

    @Override
    public Struct getSourceType() {
      return sourceType;
    }

    @Override
    public Tuple apply(Tuple t) {
      return (Tuple)coercer.apply(t);
    }

  }

  private final Struct sourceType;
  private final Struct producedType;
  private final Coercer coercer;

  @Override
  public ResultOrProblems<Projector> getProjectionFunction(Struct forSourceType) {
    assert(forSourceType == coercer.getSourceType());
    return ResultOrProblems.of(new CoercerProjector());
  }

}
