/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.plugin;

import java.lang.reflect.Constructor;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Collections;
import java.util.Set;

import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import nz.org.riskscape.engine.BuildInfo;

/**
 * A plugin descriptor for code that wasn't dynamically loaded (and can also be used in some testing scenarios)
 */
@RequiredArgsConstructor @EqualsAndHashCode(of= "pluginId")
public class BuiltInPluginDescriptor implements PluginDescriptor {

  /**
   * Build a plugin for use in tests, setting the source path to be the process working directory.
   */
  public static <T extends Plugin> T testPlugin(Class<T> pluginClass) {
    return testPlugin(pluginClass, Paths.get("."));
  }

  /**
   * Build a plugin for use in tests
   */
  public static <T extends Plugin> T testPlugin(String pluginId, Class<T> pluginClass, Path sourcePath) {
    BuiltInPluginDescriptor pluginDescriptor =
        new BuiltInPluginDescriptor(pluginId, BuildInfo.UNKNOWN, pluginClass, sourcePath);

    return pluginClass.cast(pluginDescriptor.newPluginInstance());
  }

  /**
   * Build a plugin for use in tests
   */
  public static <T extends Plugin> T testPlugin(Class<T> pluginClass, Path sourcePath) {
    return testPlugin(pluginClass.getSimpleName(), pluginClass, sourcePath);
  }

  @Getter
  private final String pluginId;
  private final BuildInfo buildInfo;
  private final Class<? extends Plugin> pluginClass;

  /**
   * The location where this plugin and its resources can be accessed.  Currently only used by the cpython plugin to
   * find the python code it needs.
   */
  @Getter
  private final Path sourcePath;

  @Override
  public ClassLoader getI18nClassLoader() {
    // NB if this returns null, the java ResourceLoader class gets upset, so we just return our class loader and
    // rely on their never being any i18n resources stored globally
    return this.getClass().getClassLoader();
  }

  @Override
  public Plugin newPluginInstance() {
    try {
      Constructor<? extends Plugin> constructor = pluginClass.getConstructor(PluginDescriptor.class);
      return constructor.newInstance(this);
    } catch (ReflectiveOperationException e) {
      throw new PluginRuntimeException(this, e, "Could not create plugin");
    }
  }

  @Override
  public boolean hasPluginDependencies() {
    return false;
  }

  @Override
  public Set<String> getPluginDependencies() {
    // plugin dependencies are class loading dependencies.  A built in plugin should therefore not have any as it uses
    // the main app classloader
    return Collections.emptySet();
  }

  @Override
  public void addDependency(PluginDescriptor dependencyDescriptor) {
    throw new UnsupportedOperationException();
  }

  @Override
  public String getPluginClassName() {
    return pluginClass.getCanonicalName();
  }

  @Override
  public String getVersion() {
    return buildInfo.getVersion();
  }

  @Override
  public String toString() {
    return String.format("BuiltInPluginDescriptor(%s %s)", getPluginId(), getPluginClassName());
  }

}
